package de.fhdw.wtf.context.model.collections;

import de.fhdw.wtf.context.model.Anything;
import de.fhdw.wtf.context.model.collections.functors.EquivalenceRelation;
import de.fhdw.wtf.context.model.collections.functors.Function;
import de.fhdw.wtf.context.model.collections.functors.Operator;
import de.fhdw.wtf.context.model.collections.functors.Predicate;

/**
 * An interface to represent an abstraction over ImmutableCollections. Every operation provides a new
 * ImmutableCollection. This empowers a functional programming style.
 * 
 * @param <T>
 *            All elements of this immutable collection have to by of type T which is a subtype to Anything.
 */
public interface ImmutableCollection<T extends Anything> extends Collection<T> {
	
	/**
	 * Provides a new Collection with the given elment added.
	 * 
	 * @param element
	 *            The newly added Element.
	 * @return Provides a new Collection with a new element added.
	 */
	ImmutableCollection<T> add(T element);
	
	/**
	 * Provides the first element of this collection.
	 * 
	 * @return The element of the front of this collection.
	 */
	T front();
	
	/**
	 * Provides a new collection without the first element.
	 * 
	 * @return Another collection without the first element of this collection.
	 */
	ImmutableCollection<T> tail();
	
	/**
	 * Provides anew collection with the elements of this collection in a reverse order.
	 * 
	 * @return Another reversed collection.
	 */
	ImmutableCollection<T> reverse();
	
	/**
	 * Applies a given function to any element of this collection and provides a collection of the results of the
	 * function.
	 * 
	 * @param <U>
	 *            The result type of the function.
	 * @param function
	 *            A unary function, which can be applied to any element of the collection.
	 * @return Provides a collection of the single results of the function.
	 */
	<U extends Anything> ImmutableCollection<U> map(Function<T, U> function);
	
	/**
	 * This method aggregates the elements of this collection to a single result by a given Aggregation Function.
	 * 
	 * @param <U>
	 *            The result type of the aggregation.
	 * @param aggregation
	 *            A binary operation +: U x T -> U.
	 * @return Provides the aggregation result of the operation.
	 */
	<U extends Anything> U reduce(Operator<T, U> aggregation);
	
	/**
	 * This method performs an aggregation of different equivalence classes of this collection. First the collection is
	 * divided into multiple sub-collections by an equivalence relation. Then these sub-collection are aggregated by the
	 * given aggregation function. The result is a collection of these results.
	 * 
	 * @param <U>
	 *            The result type of the aggregation.
	 * @param equi
	 *            An equivalence relation which divides this collection into smaller parts.
	 * @param aggregation
	 *            A binary operation +: U x T -> U.
	 * @return A Collection of the results of the different aggregations.
	 */
	<U extends Anything> ImmutableCollection<U> reduceEquivalent(EquivalenceRelation<T> equi, Operator<T, U> aggregation);
	
	/**
	 * Provides a new Collection, where each element of the new collection suffices the predicate.
	 * 
	 * @param predicate
	 *            A predicate for an element T.
	 * @return Produces a new collection with elements of this collection, which suffice the predicate.
	 */
	ImmutableCollection<T> filter(Predicate<T> predicate);
	
	/**
	 * This operation searches for the first element of this collection, which suffices the Predicate. If no element is
	 * found, a NoSuchElementException is thrown.
	 * 
	 * @param predicate
	 *            A predicate for an element of T.
	 * @return Provides the first found element which suffices the predicate.
	 */
	T find(Predicate<T> predicate);
	
}
