package de.fhdw.wtf.persistence.meta;

/**
 * A class to represent an association between an owning user type and a object.
 * 
 */
public abstract class Association extends ModelItem {
	
	/**
	 * The Owning side of the association.
	 */
	private final UserType owner;
	
	/**
	 * 
	 * The target side of the association.
	 */
	private final Type target;
	
	/**
	 * A boolean flag, indicating whether a modification of the objects, which are instances of this association should
	 * be affecting the owner of this association.
	 */
	private final boolean essential;
	
	/**
	 * A boolean flag indicating whether this association is set-valued. If it is false it is possible to have more than
	 * links, which are instance of this and have the same owner.
	 */
	private final boolean unique;
	
	/**
	 * Getter for essential.
	 * 
	 * @return Provides true whether this association has an essential flag.
	 */
	public boolean isEssential() {
		return this.essential;
	}
	
	/**
	 * Getter for unique.
	 * 
	 * @return Provides true whether this association has an unique flag.
	 */
	public boolean isUnique() {
		return this.unique;
	}
	
	/**
	 * Getter for the owner of the association.
	 * 
	 * @return Provides the owner type of the association.
	 */
	public UserType getOwner() {
		return this.owner;
	}
	
	/**
	 * Getter for the target of the association.
	 * 
	 * @return Provides the target type of the association.
	 */
	public Type getTarget() {
		return this.target;
	}
	
	/**
	 * Constructor for a new association. It should only be calles by the class facade or unit tests because the Ids are
	 * provided by the database.
	 * 
	 * @param id
	 *            The Id of the association in the database.
	 * @param name
	 *            The name of the UnidirectionalAssociation.
	 * @param owner
	 *            The Owner type of the association.
	 * @param target
	 *            The target type of the association.
	 * @param essential
	 *            A flag indicating whether this association essential-
	 * @param unique
	 *            A flag indicating whether the association is set-valued or not.
	 */
	public Association(final long id,
			final String name,
			final UserType owner,
			final Type target,
			final boolean essential,
			final boolean unique) {
		super(id, name);
		this.owner = owner;
		this.target = target;
		this.essential = essential;
		this.unique = unique;
	}
	
	@Override
	public boolean isTheSameAs(final java.lang.Object other) {
		if (!(other instanceof Association)) {
			return false;
		}
		
		final Association asso = (Association) other;
		
		// id because owner/target(/key) takes to long time and the id is unique for all associations.
		return this.getId() == asso.getId();
	}
	
}
