package de.fhdw.wtf.persistence.facade;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import de.fhdw.wtf.persistence.exception.ClassFacadeUninitializedException;
import de.fhdw.wtf.persistence.exception.FileDeleteIOException;
import de.fhdw.wtf.persistence.exception.PersistenceException;
import de.fhdw.wtf.persistence.exception.RuntimePersistenceException;
import de.fhdw.wtf.persistence.meta.Association;
import de.fhdw.wtf.persistence.meta.MapAssociation;
import de.fhdw.wtf.persistence.meta.Type;
import de.fhdw.wtf.persistence.meta.UnidirectionalAssociation;
import de.fhdw.wtf.persistence.meta.UserType;

/**
 * This class implements a classfacade which works not directly on the database for altering operations on an existing
 * model. All operations for model altering will be written to a pl-sql script which can be executed asynchronously on
 * the real database later with a sql plus client. Operations which are unprovided for the model altering, i.e.
 * initialization or clear routines, will be delegated and executed on the next instance of the classfacade-chain. You
 * can prefix an existing classfacade implementation with this one, if you wish to use the asynchronous altering
 * ability.
 * 
 */
public class AsynchronousAlteringOnExistingModellClassFacadeImplementation implements ClassFacade {
	private ClassFacade instanceToDelegateNonAlteringCalls = null;
	/**
	 * The name of the Script file.
	 */
	public static final String SCRIPT_FILE_NAME = "modelChangesScript.sql";
	
	/**
	 * Creates a new script file if it not already exists.
	 */
	private void createScriptFileIfNotExists() throws IOException {
		final File fileScript = new File(SCRIPT_FILE_NAME);
		if (!fileScript.exists()) {
			fileScript.createNewFile();
		}
	}
	
	/**
	 * Deletes the current change script if it already exists.
	 * 
	 * @throws IOException
	 *             will be thrown, when the file cannot be deleted.
	 * */
	public static void deleteScriptFileIfExists() throws IOException {
		final File fileScript = new File(SCRIPT_FILE_NAME);
		if (fileScript.exists() && !fileScript.isDirectory()) {
			if (!fileScript.delete()) {
				throw new FileDeleteIOException(fileScript);
			}
		}
	}
	
	public AsynchronousAlteringOnExistingModellClassFacadeImplementation(final ClassFacade instanceToDelegateNonAlteringCalls) {
		this.instanceToDelegateNonAlteringCalls = instanceToDelegateNonAlteringCalls;
	}
	
	@Override
	public UserType createUserType(final String name, final boolean abs, final boolean transaction)
			throws PersistenceException {
		
		final String theNewId = Long.toString(IDManager.instance().pullNextUnusedTypeID(name));
		final String absInt = abs ? "1" : "0";
		final String transInt = transaction ? "1" : "0";
		final String createUserTypeCommand =
				"execute " + OracleDatabaseManager.getInstance().getSchemaName() + ".classfacade.createUserType("
						+ theNewId + "," + "'" + name + "'" + "," + absInt + "," + transInt + ");";
		
		try {
			this.createScriptFileIfNotExists();
		} catch (final IOException e) {
			throw new RuntimePersistenceException(e);
		}
		
		try (final PrintWriter scriptWriter =
				new PrintWriter(new BufferedWriter(new FileWriter(SCRIPT_FILE_NAME, true)))) {
			scriptWriter.println(createUserTypeCommand);
			scriptWriter.flush();
		} catch (final IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		
		return new UserType(Long.parseLong(theNewId), name, abs, transaction);
	}
	
	@Override
	public UnidirectionalAssociation createUnidirectionalAssociation(final String name,
			final boolean essential,
			final boolean unique,
			final UserType owner,
			final Type target) throws PersistenceException {
		final String essentialInt = essential ? "1" : "0";
		final String uniqueInt = unique ? "1" : "0";
		final String theNewId = Long.toString(IDManager.instance().pullNextUnusedAssociationID(name));
		final String createAssociationCommand =
				"execute " + OracleDatabaseManager.getInstance().getSchemaName() + ".classfacade.createAssociation("
						+ theNewId + "," + "'" + name + "'" + "," + Long.toString(owner.getId()) + ","
						+ Long.toString(target.getId()) + "," + essentialInt + "," + uniqueInt + ");";
		
		try {
			this.createScriptFileIfNotExists();
		} catch (final IOException e) {
			throw new RuntimePersistenceException(e);
		}
		
		try (final PrintWriter scriptWriter =
				new PrintWriter(new BufferedWriter(new FileWriter(SCRIPT_FILE_NAME, true)))) {
			scriptWriter.println(createAssociationCommand);
			scriptWriter.flush();
		} catch (final IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		
		return new UnidirectionalAssociation(Long.parseLong(theNewId), name, owner, target, essential, unique);
	}
	
	@Override
	public MapAssociation createMapAssociation(final String name,
			final boolean essential,
			final UserType owner,
			final Type target,
			final Type keyType) throws PersistenceException {
		final String essentialInt = essential ? "1" : "0";
		final String uniqueInt = "1";
		final String theNewId = Long.toString(IDManager.instance().pullNextUnusedAssociationID(name));
		final String createAssociationThreeDigitCommand =
				"execute " + OracleDatabaseManager.getInstance().getSchemaName() + ".classfacade.createAssociation3("
						+ theNewId + "," + "'" + name + "'" + "," + Long.toString(owner.getId()) + ","
						+ Long.toString(target.getId()) + "," + essentialInt + "," + uniqueInt + ","
						+ Long.toString(keyType.getId()) + ");";
		
		try {
			this.createScriptFileIfNotExists();
		} catch (final IOException e) {
			throw new RuntimePersistenceException(e);
		}
		
		try (final PrintWriter scriptWriter =
				new PrintWriter(new BufferedWriter(new FileWriter(SCRIPT_FILE_NAME, true)))) {
			scriptWriter.println(createAssociationThreeDigitCommand);
			scriptWriter.flush();
		} catch (final IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		
		return new MapAssociation(Long.parseLong(theNewId), name, owner, target, keyType, essential);
	}
	
	@Override
	public void createSpecializationBetween(final UserType ancestor, final Type descendant) throws PersistenceException {
		
		final String createSpecializationCommand =
				"execute " + OracleDatabaseManager.getInstance().getSchemaName() + ".classfacade.createSpecialization("
						+ Long.toString(ancestor.getId()) + "," + Long.toString(descendant.getId()) + ");";
		
		try {
			this.createScriptFileIfNotExists();
		} catch (final IOException e) {
			throw new RuntimePersistenceException(e);
		}
		
		try (final PrintWriter scriptWriter =
				new PrintWriter(new BufferedWriter(new FileWriter(SCRIPT_FILE_NAME, true)))) {
			scriptWriter.println(createSpecializationCommand);
			scriptWriter.flush();
		} catch (final IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
	}
	
	@Override
	public boolean isSuperClassTo(final Type ancestor, final Type descendant) throws PersistenceException {
		
		return this.instanceToDelegateNonAlteringCalls.isSuperClassTo(ancestor, descendant);
	}
	
	@Override
	public void finalizeSpecialization() throws PersistenceException {
		
		final String finalizeSpecializationCommand =
				"execute " + OracleDatabaseManager.getInstance().getSchemaName()
						+ ".classfacade.finalizeSpecialization;";
		
		try {
			this.createScriptFileIfNotExists();
		} catch (final IOException e) {
			throw new RuntimePersistenceException(e);
		}
		
		try (final PrintWriter scriptWriter =
				new PrintWriter(new BufferedWriter(new FileWriter(SCRIPT_FILE_NAME, true)))) {
			scriptWriter.println(finalizeSpecializationCommand);
			scriptWriter.flush();
		} catch (final IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
	}
	
	@Override
	public void clear() throws PersistenceException {
		this.instanceToDelegateNonAlteringCalls.clear();
		
	}
	
	@Override
	public void initialize() throws PersistenceException {
		this.instanceToDelegateNonAlteringCalls.clear();
		
	}
	
	@Override
	public void initializeForRuntime() throws PersistenceException {
		this.instanceToDelegateNonAlteringCalls.initializeForRuntime();
	}
	
	@Override
	public boolean hasBeenInitialized() {
		return this.instanceToDelegateNonAlteringCalls.hasBeenInitialized();
	}
	
	@Override
	public TypeManager getTypeManager() throws ClassFacadeUninitializedException {
		return this.instanceToDelegateNonAlteringCalls.getTypeManager();
	}
	
	@Override
	public void renameType(final Long typeId, final String newName) throws PersistenceException {
		String command =
				"execute " + OracleDatabaseManager.getInstance().getSchemaName() + ".classfacade.renameType(%s,%s);";
		command = String.format(command, typeId.toString(), newName);
		this.writeCommandToFile(command);
	}
	
	@Override
	public void renameAssociation(final Long assoId, final String newName) throws PersistenceException {
		String command =
				"execute " + OracleDatabaseManager.getInstance().getSchemaName()
						+ ".classfacade.renameAssociation(%s,%s);";
		command = String.format(command, assoId.toString(), newName);
		this.writeCommandToFile(command);
	}
	
	@Override
	public void deleteAssociation(final Long associationId) throws PersistenceException {
		String command =
				"execute " + OracleDatabaseManager.getInstance().getSchemaName()
						+ ".classfacade.deleteAssociation(%s);";
		command = String.format(command, associationId.toString());
		this.writeCommandToFile(command);
	}
	
	@Override
	public void deleteUserType(final Long typeId) throws PersistenceException {
		String command =
				"execute " + OracleDatabaseManager.getInstance().getSchemaName()
						+ ".classfacade.deleteUserTypeAndSpec(%s);";
		command = String.format(command, typeId.toString());
		this.writeCommandToFile(command);
	}
	
	@Override
	public void updateLinksToNewAssociation(final Long associationId, final Collection<Long> newAssociationIds)
			throws PersistenceException {
		String command =
				"execute " + OracleDatabaseManager.getInstance().getSchemaName()
						+ ".classfacade.pushDownLinks(%s,ARRAY_INT(%s));";
		command = String.format(command, associationId.toString(), this.createCSVFormat(newAssociationIds));
		this.writeCommandToFile(command);
	}
	
	@Override
	public void moveLinksAndCreateObjects(final List<Long> oldAssoIds,
			final Association newAsso,
			final UserType newType,
			final List<Long> newAssoIds) throws PersistenceException {
		String command =
				"execute " + OracleDatabaseManager.getInstance().getSchemaName()
						+ ".classfacade.moveLinksAndCreateObjects(ARRAY_INT(%s),%s,%s,ARRAY_INT(%s));";
		command =
				String.format(command, this.createCSVFormat(oldAssoIds), Long.valueOf(newAsso.getId()).toString(), Long
						.valueOf(newType.getId()).toString(), this.createCSVFormat(newAssoIds));
		this.writeCommandToFile(command);
	}
	
	private String createCSVFormat(final Collection<Long> newAssociationIds) {
		final StringBuffer resultBuffer = new StringBuffer();
		final Iterator<Long> i = newAssociationIds.iterator();
		if (i.hasNext()) {
			resultBuffer.append(i.next().toString());
		}
		while (i.hasNext()) {
			final Long current = i.next();
			resultBuffer.append(",");
			resultBuffer.append(current.toString());
		}
		return resultBuffer.toString();
	}
	
	private void writeCommandToFile(final String command) {
		try {
			this.createScriptFileIfNotExists();
		} catch (final IOException e) {
			throw new RuntimePersistenceException(e);
		}
		
		try (final PrintWriter scriptWriter =
				new PrintWriter(new BufferedWriter(new FileWriter(SCRIPT_FILE_NAME, true)))) {
			scriptWriter.println(command);
			scriptWriter.flush();
		} catch (final IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
	}
	
}
