CREATE OR REPLACE PACKAGE BODY objectfacade IS

--Helper functions

/*  diese Funktion liefert die ID der aktiven Transaktion fr die
    bergebene lange Transaktion.
*/
FUNCTION getActiveShortTransaction(trans NUMBER) RETURN NUMBER IS
	result NUMBER;
	cur TransactionIDCollection;
BEGIN
	OPEN cur FOR 
		SELECT short_transaction.id INTO result FROM short_transaction
		INNER JOIN long_transaction ON long_transaction.id=short_transaction.container
		WHERE long_transaction.id=trans
		AND short_transaction.next IS NULL;

	FETCH cur INTO result;
	IF cur%NOTFOUND THEN
		result := short_transaction_seq.Nextval;
		INSERT INTO short_transaction VALUES (result, trans, NULL);
	END IF;
	CLOSE cur;
RETURN result;
END getActiveShortTransaction;

/*
	Die Funktion userObjectCreatedBy liefert die Id der langen Transaktion, die das UserObject mit der Id <userObject> erstellt hat.
*/
FUNCTION userObjectCreatedBy(userObject NUMBER) RETURN NUMBER IS
	creatorLTA NUMBER;
BEGIN
	SELECT lt.id INTO creatorLTA FROM user_object uo 
		INNER JOIN short_transaction st ON uo.created=st.id
		INNER JOIN Long_transaction lt ON st.container=lt.id
	WHERE  uo.id=userObject;
	RETURN creatorLTA;
	EXCEPTION
		WHEN NO_DATA_FOUND THEN
		creatorLTA := NULL;
	RETURN creatorLTA;
END userObjectCreatedBy;
 
/*
	Die Funktion linkCreatedBy liefert die Id der langen Transaktion, die den Link mit der Id <link> erstellt hat.
*/
FUNCTION linkCreatedBy(link NUMBER) RETURN NUMBER IS
	creatorLTA NUMBER;
BEGIN
	SELECT lt.id INTO creatorLTA FROM Link l 
		INNER JOIN short_transaction st ON l.created=st.id
		INNER JOIN Long_transaction lt ON st.container=lt.id
	WHERE l.id=link;
	RETURN creatorLTA;
	EXCEPTION
		WHEN NO_DATA_FOUND THEN
		creatorLTA := NULL;
	RETURN creatorLTA;
END linkCreatedBy;

/*
	Die Funktion wasVersionedObjectCreatedBy liefert True, wenn die lange Transaktion mit der Id <lt> das UserObject bzw. den Link mit der Id <versionedObject> erstellt hat.
*/
FUNCTION wasVersionedObjectCreatedBy(versionedObject NUMBER, lt NUMBER) RETURN BOOLEAN IS
	result BOOLEAN;
BEGIN
	result := linkCreatedBy(versionedObject)=lt;
	IF result IS NULL THEN
		result := userObjectCreatedBy(versionedObject)=lt;
	END IF;
RETURN result;
END;

/*
	Die Funktion insertReadAccessed fgt ein gelesen-Accessed Vermerk fr das UserObject bzw. den Link mit der Id <versionedObject> ein.
	Wenn die lange Transaktion mit der Id <trans> das UserObject bzw. den Link mit der Id <versionedObject> erstellt hat, dann wird kein Vermerk eingefgt.
	Wenn bereits ein Vermerk fr das UserObject bzw. den Link mit der Id <versionedObject> IN der langen Transaktion mit der Id <trans> besteht, dann wird kein Vermerk eingefgt
*/
PROCEDURE insertReadAccessed(versionedObject NUMBER, trans NUMBER)  IS 
	accessedCount NUMBER;
	st NUMBER;
BEGIN
	IF NOT wasVersionedObjectCreatedBy(versionedObject, trans) THEN
		st := getActiveShortTransaction(trans);
		SELECT COUNT(*) INTO accessedCount FROM accessed
			INNER JOIN short_transaction ON short_transaction.id=accessed.t
			INNER JOIN long_transaction ON long_transaction.id=short_transaction.container
		WHERE accessed.v=versionedObject
			AND long_transaction.id=trans;
		IF accessedCount=0 THEN
		  INSERT INTO accessed VALUES (Accessed_seq.nextval, versionedObject, st, 'R');
		END IF;
	END IF;
END insertReadAccessed;

/*
	Die Funktion insertModifiedAccessed fgt ein modifiziert-Accessed Vermerk fr das UserObject bzw. den Link mit der Id <versionedObject> ein.
	Wenn die lange Transaktion mit der Id <trans> das UserObject bzw. den Link mit der Id <versionedObject> erstellt hat, dann wird kein Vermerk eingefgt.
	Wenn bereits ein gelscht-Accessed Vermerk oder ein modifiziert-Accessed Vermerk fr das UserObject bzw. den Link mit der Id <versionedObject>
	in der langen Transaktion mit der Id <trans> besteht, dann wird kein Vermerk eingefgt
	Wenn bereits ein gelesen-Accessed Vermerkt fr das UserObject bzw. den Link mit der Id <versionedObject>
	in der langen Transaktion mit der Id <trans> besteht, dann wird er auf ein modifiziert-Accessed Vermerk angehoben.
*/  
PROCEDURE insertModifiedAccessed(versionedObject NUMBER, trans NUMBER)  IS 
	deletedCount NUMBER;
	modifiedCount NUMBER;
	readCount NUMBER;
	st NUMBER;
BEGIN
	IF NOT wasVersionedObjectCreatedBy(versionedObject, trans) THEN
		st := getActiveShortTransaction(trans);
		SELECT COUNT(*) INTO deletedCount FROM accessed
			INNER JOIN short_transaction ON short_transaction.id=accessed.t AND short_transaction.container=trans
		WHERE accessed.v=versionedObject AND accessed.type='D';
		IF deletedCount=0 THEN
			SELECT COUNT(*) INTO modifiedCount FROM accessed
				INNER JOIN short_transaction ON short_transaction.id=accessed.t AND short_transaction.container=trans
			WHERE accessed.v=versionedObject AND accessed.type='M';      
			IF modifiedCount=0 THEN
				SELECT COUNT(*) INTO readCount FROM accessed
					INNER JOIN short_transaction ON short_transaction.id=accessed.t  AND short_transaction.container=trans
				WHERE accessed.v=versionedObject AND accessed.type='R';
				IF readCount=0 THEN
					INSERT INTO accessed VALUES (Accessed_seq.nextval, versionedObject, st, 'M');
				ELSE
					UPDATE accessed SET type='M' 
					WHERE accessed.v=versionedObject AND accessed.t IN 
					(SELECT short_transaction.id FROM accessed
						INNER JOIN short_transaction ON short_transaction.id=accessed.t
					WHERE short_transaction.container=trans AND accessed.v=versionedObject AND accessed.type='R');
				END IF;
			END IF;
		END IF;
	END IF;
END insertModifiedAccessed;
  
/*
	Die Funktion insertDeletedAccessed fgt ein gelscht-Accessed Vermerk fr das UserObject bzw. den Link mit der Id <versionedObject> ein.
	Wenn die lange Transaktion mit der Id <trans> das UserObject bzw. den Link mit der Id <versionedObject> erstellt hat, dann wird kein Vermerk eingefgt.
	Wenn bereits ein gelscht-Accessed Vermerk fr das UserObject bzw. den Link mit der Id <versionedObject>
	in der langen Transaktion mit der Id <trans> besteht, dann wird kein Vermerk eingefgt
	Wenn bereits ein gelesen-Accessed Vermerkt oder ein modifiziert-Accessed Vermerk fr das UserObject bzw. den Link mit der Id <versionedObject>
	in der langen Transaktion mit der Id <trans> besteht, dann wird er auf ein gelscht-Accessed Vermerk angehoben.
*/  
PROCEDURE insertDeletedAccessed(versionedObject NUMBER, trans NUMBER) IS 
	deletedCount NUMBER;
	st NUMBER;
BEGIN
	IF NOT wasVersionedObjectCreatedBy(versionedObject, trans) THEN
		st := getActiveShortTransaction(trans);
		SELECT COUNT(*) INTO deletedCount FROM accessed
			INNER JOIN short_transaction ON short_transaction.id=accessed.t AND short_transaction.container=trans
		WHERE accessed.v=versionedObject AND accessed.type='D';
		IF deletedCount=0 THEN
			INSERT INTO accessed VALUES (Accessed_seq.nextval, versionedObject, st, 'D');
		ELSE
			UPDATE accessed SET type='D'
			WHERE accessed.v=versionedObject AND accessed.t IN
			(SELECT short_transaction.id FROM short_transaction WHERE short_transaction.container=trans);
		END IF;
	END IF;
END insertDeletedAccessed;


FUNCTION isMapAssociation(a NUMBER) RETURN BOOLEAN IS
	isMapAssociation NUMBER;
BEGIN
	isMapAssociation := 0;
	SELECT count(*) INTO isMapAssociation FROM MapAssociation 
		WHERE id=a;
	IF isMapAssociation != 0 THEN
		RETURN TRUE;
	ELSE
		RETURN FALSE;
	END IF;
END;

/* isUserObject(o NUMBER) RETURN BOOLEAN 
Die Hilfsfunktion isUserObject() berprft ob die angegebene ID <o> ein Userobjekt ist und gibt bei erfolg TRUE zurck, Anderenfalls FALSE */
FUNCTION isUserObject(o NUMBER) RETURN BOOLEAN IS
	isUserObject NUMBER;
BEGIN
	isUserObject := 0;
	SELECT count(*) INTO isUserObject FROM USER_OBJECT
		WHERE ID=o;
	IF isUserObject != 0 THEN
		RETURN TRUE;
	ELSE
		RETURN FALSE;
	END IF;
END;

/* isUnique(assoc NUMBER) RETURN BOOLEAN
   Die Hilfsoperation isUnique() bekommt eine ID zu der Assoziation <assoc> bergeben und prft, ob diese als unique markiert ist.
   Liefert true falls <assoc> unique war, sonst false.
*/
FUNCTION isUnique(assoc NUMBER) RETURN BOOLEAN IS
	isUniqueSet NUMBER;
BEGIN
	isUniqueSet := 0;
	SELECT count(*) INTO isUniqueSet FROM Association
		WHERE ID=assoc AND UNIQ=1;
	IF isUniqueSet != 0 THEN
		RETURN TRUE;
	ELSE
		RETURN FALSE;
	END IF;
END;

/* isEssential(assoc NUMBER) RETURN BOOLEAN
   Die Hilfsoperation isEssential() bekommt eine ID zu der Assoziation <assoc> bergeben und prft, ob diese als essential markiert ist.
   Liefert true falls <assoc> essential war, sonst false.
*/
FUNCTION isEssential(assoc NUMBER) RETURN BOOLEAN IS
	essentialIsSet NUMBER;
BEGIN
	SELECT count(*) INTO essentialIsSet FROM Association
		WHERE ID=assoc AND ESSENTIAL=1;
	IF essentialIsSet != 0 THEN
		RETURN TRUE;
	ELSE
		RETURN FALSE;
	END IF;
END;

/* isSpecializationOfOwner(a NUMBER, o NUMBER) RETURN BOOLEAN
   Die Hilfsoperation isSpecializationOfOwner prueft, ob das User_Object <o> von einem spezielleren oder gleich dem Typ des owners von <a> ist.
   Liefert true, falls der Typ von <o> gleich oder spezieller ist als der owner von <a>.
*/
FUNCTION isSpecializationOfOwner(a NUMBER, o NUMBER) RETURN BOOLEAN IS
	owner_io NUMBER;
	owner_o_io NUMBER;
	isSuperClassVal NUMBER;
BEGIN
	SELECT owner INTO owner_io FROM Association
		WHERE id=a;
	SELECT instance_of INTO owner_o_io FROM object
		WHERE id=o;
	SELECT classfacade.isSuperclassTo(owner_io,owner_o_io) INTO isSuperClassVal FROM dual;
	IF isSuperClassVal != 0 THEN
		RETURN TRUE;
	ELSE
		RETURN FALSE;
	END IF;
END;

/* isSpecializationOfTarget(a NUMBER, tParam NUMBER) RETURN BOOLEAN
   Die Hilfsoperation isSpecializationOfTarget prueft, ob das User_Object <o> von einem spezielleren oder gleich dem Typ des target von <a> ist.
   Liefert true, falls der Typ von <tParam> gleich oder spezieller ist als der target von <a>.
*/
FUNCTION isSpecializationOfTarget(a NUMBER, tParam NUMBER) RETURN BOOLEAN IS
	target_io NUMBER;
	target_o_io NUMBER;
	isSuperClassVal NUMBER;
BEGIN
	SELECT target INTO target_io FROM Association
		WHERE id=a;
	SELECT instance_of INTO target_o_io FROM object
		WHERE id=tParam;
	SELECT classfacade.isSuperclassTo(target_io,target_o_io) INTO isSuperClassVal FROM dual;
	IF isSuperClassVal != 0 THEN
		RETURN TRUE;
	ELSE
		RETURN FALSE;
	END IF;
END;

/* isSpecializationOfKey(a NUMBER, k NUMBER) RETURN BOOLEAN
   Die Hilfsoperation isSpecializationOfKey prueft, ob das User_Object <o> von einem spezielleren oder gleich dem Typ des keys von <a> ist.
   Liefert true, falls der Typ von <k> gleich oder spezieller ist als der key von <a>.
*/
FUNCTION isSpecializationOfKey(a NUMBER, k NUMBER) RETURN BOOLEAN IS
	key_io NUMBER;
	key_o_io NUMBER;
	isSuperClassVal NUMBER;
BEGIN
	SELECT map_key_type INTO key_io FROM MapAssociation
		WHERE id=a;
	SELECT instance_of INTO key_o_io FROM object
		WHERE id=k;
	SELECT classfacade.isSuperclassTo(key_io,key_o_io) INTO isSuperClassVal FROM dual;
	IF isSuperClassVal != 0 THEN
		RETURN TRUE;
	ELSE
		RETURN FALSE;
	END IF;
END;

/* isUOInScope(uo NUMBER, trans NUMBER) RETURN BOOLEAN
   Die Hilfsoperation isUOInScope prueft, ob das User_Object <uo> IN der Transaktion <trans> sichtbar ist.
   Liefert true, falls es sichtbar ist.
*/
FUNCTION isUOInScope(uo NUMBER, trans NUMBER) RETURN BOOLEAN IS
	oIsVisibleInTransactionContext NUMBER;
BEGIN
	SELECT count(*) INTO oIsVisibleInTransactionContext FROM (SELECT * FROM USER_OBJECT WHERE ID=uo) userObject1
		INNER JOIN SHORT_TRANSACTION st1 ON st1.id=userobject1.created
		INNER JOIN (SELECT * FROM LONG_TRANSACTION WHERE rightDateGreaterEqualsLeftDate(PUBLISHED, SYSDATE)=1 OR ID=trans) longTransAction1 ON longTransAction1.ID=st1.container
	WHERE userObject1.ID NOT IN 
	(SELECT ac1.V FROM ACCESSED ac1
		INNER JOIN SHORT_TRANSACTION st2 ON ac1.T=st2.ID
		INNER JOIN LONG_TRANSACTION lt2 ON lt2.ID=st2.CONTAINER
	WHERE (ac1.TYPE='D' AND (rightDateGreaterEqualsLeftDate(lt2.PUBLISHED, SYSDATE)=1 OR lt2.ID=trans)));
	IF oIsVisibleInTransactionContext != 0 THEN
		RETURN TRUE;
	ELSE
		RETURN FALSE;
	END IF;
END;

/* isLinkInScope(linkID NUMBER, trans NUMBER) RETURN BOOLEAN
   Die Hilfsoperation isLinkInScope prueft, ob der Link <linkID> IN der Transaktion <trans> sichtbar ist.
   Liefert true, falls der Link sichtbar ist.
*/
FUNCTION isLinkInScope(linkID NUMBER, trans NUMBER) RETURN BOOLEAN IS
	oIsVisibleInTransactionContext NUMBER;
BEGIN
	SELECT count(*) INTO oIsVisibleInTransactionContext FROM (SELECT * FROM Link WHERE ID=linkID) link1
		INNER JOIN SHORT_TRANSACTION st1 ON st1.id=link1.created
		INNER JOIN (SELECT * FROM LONG_TRANSACTION WHERE rightDateGreaterEqualsLeftDate(PUBLISHED, SYSDATE)=1 OR ID=trans) longTransAction1 ON longTransAction1.ID=st1.container
	WHERE link1.ID NOT IN 
		(SELECT ac1.V FROM ACCESSED ac1
			INNER JOIN SHORT_TRANSACTION st2 ON ac1.T=st2.ID
			INNER JOIN LONG_TRANSACTION lt2 ON lt2.ID=st2.CONTAINER
		WHERE (ac1.TYPE='D' AND (rightDateGreaterEqualsLeftDate(lt2.PUBLISHED, SYSDATE)=1 OR lt2.ID=trans)));
	IF oIsVisibleInTransactionContext != 0 THEN
		RETURN TRUE;
	ELSE
		RETURN FALSE;
	END IF;
END;

/* getLinkCountOfAssoForOwner(trans NUMBER, o NUMBER, a NUMBER) RETURN NUMBER
   Diese Hilfsoperation liefert die Anzahl an Links vom Typ <a>, welche innerhalb der Transaktion <trans> fr das User_Object <o> sichtbar sind. 
*/
FUNCTION getLinkCountOfAssoForOwner(trans NUMBER, o NUMBER, a NUMBER) RETURN NUMBER IS
	linkAmount NUMBER;
BEGIN
	SELECT count(*) INTO linkAmount FROM (SELECT * FROM Link WHERE owner=o AND INSTANCE_OF=a) possibleLinks1
		INNER JOIN SHORT_TRANSACTION st1 ON st1.ID=possibleLinks1.CREATED
		INNER JOIN (SELECT * FROM LONG_TRANSACTION WHERE rightDateGreaterEqualsLeftDate(PUBLISHED, SYSDATE)=1 OR ID=trans) lt1 ON lt1.ID=st1.CONTAINER
	WHERE possibleLinks1.ID NOT IN 
		(SELECT ac1.V FROM ACCESSED ac1
			INNER JOIN SHORT_TRANSACTION st2 ON ac1.T=st2.ID
			INNER JOIN LONG_TRANSACTION lt2 ON lt2.ID=st2.CONTAINER
		WHERE (ac1.TYPE='D' AND (rightDateGreaterEqualsLeftDate(lt2.PUBLISHED, SYSDATE)=1 OR lt2.ID=trans)));
	RETURN linkAmount;
END;

/* getLinkCountOfAssoForOwnerK(trans NUMBER, o NUMBER, k NUMBER, a NUMBER)
   Diese Hilfsoperation liefert die Anzahl an Links vom Typ <a>, welche innerhalb der Transaktion <trans> fr das User_Object <o> und den Key <k> sichtbar sind. 
*/
FUNCTION getLinkCountOfAssoForOwnerK(trans NUMBER, o NUMBER, k NUMBER, a NUMBER) RETURN NUMBER IS
	linkAmount NUMBER;
BEGIN
	SELECT count(*) INTO linkAmount FROM (SELECT Link.* FROM Link INNER JOIN MapLink ON MapLink.id=Link.id WHERE owner=o AND INSTANCE_OF=a AND MapLink.map_key=k) possibleLinks1
		INNER JOIN SHORT_TRANSACTION st1 ON st1.ID=possibleLinks1.CREATED
		INNER JOIN (SELECT * FROM LONG_TRANSACTION WHERE rightDateGreaterEqualsLeftDate(PUBLISHED, SYSDATE)=1 OR ID=trans) lt1 ON lt1.ID=st1.CONTAINER
	WHERE possibleLinks1.ID NOT IN 
		(SELECT ac1.V FROM ACCESSED ac1
			INNER JOIN SHORT_TRANSACTION st2 ON ac1.T=st2.ID
			INNER JOIN LONG_TRANSACTION lt2 ON lt2.ID=st2.CONTAINER
		WHERE (ac1.TYPE='D' AND (rightDateGreaterEqualsLeftDate(lt2.PUBLISHED, SYSDATE)=1 OR lt2.ID=trans)));
	RETURN linkAmount;
END;

/* getUniqueLinkIdOfAssoForOwner(lt NUMBER, o NUMBER, a NUMBER) RETURN NUMBER
   Diese Hilfsoperation liefert die ID des einzigen links welches innerhalb der Transaktion <lt> ist, von <o> besessen wird und durch <a> typisiert ist.
   Eine Pruefung auf die Existenz eines solchen links muss vor dem Aufruf dieser Operation erfolgen z.B. durch getLinkCountOfAssoForOwner().
*/
FUNCTION getUniqueLinkIdOfAssoForOwner(lt NUMBER, o NUMBER, a NUMBER) RETURN NUMBER IS
	returnID NUMBER;
BEGIN
	SELECT ID INTO returnID FROM Link 
		WHERE owner=o AND INSTANCE_OF=a AND getLinkCountOfAssoForOwner(lt, o, a) != 0;
	RETURN returnID;
END;

/* getUniqueLinkIdOfAssoForOwnerK(lt NUMBER, o NUMBER, k NUMBER, a NUMBER) RETURN NUMBER
   Diese Hilfsoperation liefert die ID des einzigen links welches innerhalb der Transaktion <lt> ist, von <o> besessen wird und durch <a> typisiert ist.
   Eine Pruefung auf die Existenz eines solchen links muss vor dem Aufruf dieser Operation erfolgen z.B. durch getLinkCountOfAssoForOwner().
*/
FUNCTION getUniqueLinkIdOfAssoForOwnerK(lt NUMBER, o NUMBER, k NUMBER, a NUMBER) RETURN NUMBER IS
	returnID NUMBER;
BEGIN
	SELECT Link.ID INTO returnID FROM Link 
		INNER JOIN MapLink ON MapLink.id=Link.id 
	WHERE owner=o AND INSTANCE_OF=a AND MapLink.map_key=k AND getLinkCountOfAssoForOwner(lt, o, a) != 0;
	RETURN returnID;
END;


/* queryFindStringByTrans(a NUMBER, s VARCHAR2, trans NUMBER) RETURN IDRecordCollection
   Diese Hilfsoperation liefert einen offenen coursor und bietet so den Zugriff auf die Menge der Objekt IDs und der typen, welche
   einen link vom Typ <a> zu dem String <s> haben und im Kontext der Transaktion <trans> sichtbar sind. 
*/
FUNCTION queryFindStringByTrans(a NUMBER, s VARCHAR2, trans NUMBER) RETURN IDRecordCollection IS
	returnValue IDRecordCollection;
BEGIN
	OPEN returnValue FOR
	SELECT DISTINCT linkValue1.owner,o2.instance_of FROM (SELECT * FROM STRING_VALUE WHERE VALUE LIKE s) stringValue1 
		INNER JOIN (SELECT * FROM Link WHERE INSTANCE_OF=a) linkValue1 ON linkValue1.target=stringValue1.ID 
		INNER JOIN USER_OBJECT userObject1 ON linkValue1.owner=userObject1.ID 
		INNER JOIN SHORT_TRANSACTION st1 ON st1.id=linkValue1.created 
		INNER JOIN (SELECT * FROM LONG_TRANSACTION 
	WHERE objectfacade.rightDateGreaterEqualsLeftDate(PUBLISHED, SYSDATE)=1 OR ID=trans) longTransAction1 ON longTransAction1.ID=st1.container 
		INNER JOIN object o2 ON o2.id=linkValue1.owner 
	WHERE linkValue1.ID NOT IN 
	(SELECT linkValue.ID FROM Link linkValue 
		INNER JOIN accessed ac1 ON ac1.V=linkValue.ID 
		INNER JOIN SHORT_TRANSACTION st2 ON ac1.T=st2.ID 
		INNER JOIN LONG_TRANSACTION lt2 ON lt2.ID=st2.CONTAINER 
	WHERE ac1.type='D' AND ((objectfacade.rightDateGreaterEqualsLeftDate(lt2.PUBLISHED, SYSDATE)=1) OR lt2.ID=trans));
	RETURN returnValue;
END;

/* queryFindIntegerByTrans(a NUMBER, i Number, trans NUMBER) RETURN IDRecordCollection
   Diese Hilfsoperation liefert einen offenen coursor und bietet so den Zugriff auf die Menge der Objekt IDs und der typen, welche
   einen link vom Typ <a> zu dem Integer <i> haben und im Kontext der Transaktion <trans> sichtbar sind. 
*/
FUNCTION queryFindIntegerByTrans(a NUMBER, i NUMBER, trans NUMBER) RETURN IDRecordCollection IS
	returnValue IDRecordCollection;
BEGIN
	OPEN returnValue FOR
	SELECT DISTINCT linkValue1.owner,o2.instance_of FROM
		(SELECT * FROM integer_value WHERE VALUE=i) integerValue1
			INNER JOIN (SELECT * FROM Link WHERE INSTANCE_OF=a) linkValue1 ON linkValue1.target=integerValue1.ID
			INNER JOIN USER_OBJECT userObject1 ON linkValue1.owner=userObject1.ID
			INNER JOIN SHORT_TRANSACTION st1 ON st1.id=linkValue1.created
			INNER JOIN 
				(SELECT * FROM LONG_TRANSACTION 
				WHERE objectfacade.rightDateGreaterEqualsLeftDate(PUBLISHED, SYSDATE)=1 OR ID=trans) longTransAction1 ON longTransAction1.ID=st1.container
			INNER JOIN object o2 ON o2.id=linkValue1.owner
		WHERE linkValue1.ID NOT IN 
			(SELECT linkValue.ID FROM Link linkValue 
				INNER JOIN accessed ac1 ON ac1.V=linkValue.ID
				INNER JOIN SHORT_TRANSACTION st2 ON ac1.T=st2.ID
				INNER JOIN LONG_TRANSACTION lt2 ON lt2.ID=st2.CONTAINER
			WHERE ac1.type='D' AND ((objectfacade.rightDateGreaterEqualsLeftDate(lt2.PUBLISHED, SYSDATE))=1) OR lt2.ID=trans);	  
	RETURN returnValue;
END;
  
/* getStringObject(s VARCHAR2) RETURN Number
   Diese Hilfsoperation preuft, ob der Wert <s> vom Typ String IN der Datenbank verfgbar ist und liefert die ID fr dieses Objekt.
   Falls kein Wert <s> verfgbar ist vom Typ String, so wird dieser angelegt und diese neue ID geliefert.
*/  
FUNCTION getStringObject(s VARCHAR2) RETURN NUMBER IS
	result NUMBER;
	stringIDExists NUMBER;
BEGIN
	SELECT count(*) INTO stringIDExists FROM STRING_VALUE WHERE VALUE=s;
	IF stringIDExists != 0 THEN
		SELECT ID INTO result FROM STRING_VALUE WHERE VALUE=s;
	ELSE 
		result := InstanceItem_Seq.NEXTVAL;
		INSERT INTO OBJECT(ID,INSTANCE_OF) VALUES(result,classfacade.string_id);
		INSERT INTO STRING_VALUE(ID,VALUE) VALUES(result,s);
	END IF;
	RETURN result;
END getStringObject;
  
/* getInteger(i NUMBER) RETURN Number
   Diese Hilfsoperation preuft, ob der Wert <i> vom Typ Integer IN der Datenbank verfgbar ist und liefert die ID fr dieses Objekt.
   Falls kein Wert <i> verfgbar ist vom Typ Integer, so wird dieser angelegt und diese neue ID geliefert.
*/
FUNCTION getIntegerObject(i NUMBER) RETURN NUMBER IS
	result NUMBER;
	intIDExists NUMBER;
BEGIN
	SELECT count(*) INTO intIDExists FROM INTEGER_VALUE WHERE VALUe=i;
	IF intIDExists != 0 THEN
		SELECT ID INTO result FROM INTEGER_VALUE WHERE VALUE=i;
	ELSE 
		result := InstanceItem_Seq.NEXTVAL;
		INSERT INTO OBJECT(ID,INSTANCE_OF) VALUES(result,classfacade.integer_id);
		INSERT INTO INTEGER_VALUE(ID,VALUE) VALUES(result,i);
	END IF;
	RETURN result;
END getIntegerObject;
  
/*  
	Die Funktion remainingLinks liefert die Anzahl der Links, die unter der
	Sicht von <longTA> <userObject> als owner oder target referenzieren.
*/
FUNCTION remainingLinks(userObject NUMBER, longTA NUMBER) RETURN NUMBER IS 
	numberOfLinks NUMBER;
BEGIN
	SELECT count(*) INTO NUMBEROfLinks from
		(SELECT l.id  FROM object o 
			INNER JOIN user_object uo ON o.id=uo.id
			INNER JOIN Link l ON l.target=uo.id OR l.owner=uo.id
			INNER JOIN short_transaction st ON st.id=l.created 
			INNER JOIN long_transaction lt ON st.container=lt.id
		WHERE o.id=userObject 
			AND (rightdategreaterequalsleftdate(lt.published, sysdate)=1)
	MINUS
	SELECT l.id FROM accessed a 
		INNER JOIN short_transaction st ON st.id=a.t
		INNER JOIN Link l ON l.id=a.v 
	WHERE a.type='D' 
		AND st.container=longTA 
		and
		(l.owner=userObject OR l.target=userObject));
	RETURN NUMBEROfLinks;
END remainingLinks;
  
/* End Helper Functions */


FUNCTION findUserObjectByType(typ NUMBER,trans NUMBER) RETURN IDRecordCollection IS
	result IDRecordCollection;
BEGIN
  OPEN result FOR
	SELECT DISTINCT object.id,object.instance_of FROM Object 
		INNER JOIN user_object ON user_object.id=object.id
		INNER JOIN short_transaction ON user_object.created=short_transaction.id
		INNER JOIN 
		(SELECT * FROM LONG_TRANSACTION 
			WHERE objectfacade.rightDateGreaterEqualsLeftDate(PUBLISHED, SYSDATE)=1 OR ID=trans) lt ON lt.id=short_transaction.container
			WHERE object.instance_of IN (with partielle_ordnung_trans(ancestor,descendant) AS
				(SELECT ancestor,descendant FROM specialization 
					WHERE ancestor=typ
				UNION ALL
				SELECT child.ancestor,parent.descendant FROM specialization child, specialization parent
					WHERE parent.ancestor=child.descendant AND parent.ancestor=typ)
			SELECT DISTINCT descendant FROM partielle_ordnung_trans p) 
		AND
		object.id NOT IN 
			(SELECT user_object.ID FROM user_object uo 
				INNER JOIN accessed ac1 ON ac1.V=uo.ID 
				INNER JOIN SHORT_TRANSACTION st2 ON ac1.T=st2.ID 
				INNER JOIN LONG_TRANSACTION lt2 ON lt2.ID=st2.CONTAINER 
			WHERE ac1.type='D' AND ((objectfacade.rightDateGreaterEqualsLeftDate(lt2.PUBLISHED, SYSDATE)=1) OR lt2.ID=trans));
  RETURN result;
END;
  
  
FUNCTION checkUserObjectOut(obj NUMBER, trans NUMBER) RETURN IDRecordCollection IS
	result IDRecordCollection;
BEGIN
    insertReadAccessed(obj,trans);
    OPEN result FOR
		SELECT id, instance_of FROM object
			WHERE id=obj;
    RETURN result;
END;
  
  
/* findString(a NUMBER, s VARCHAR2, d TIMESTAMP)IDRecordCollection
   Diese Operation realisiert die Suche nach den User_Object-Identifikatoren, welche am Datum <d> einen Link vom Typ <a> auf ein String mit dem Wert <s> haben.
   Die entsprechende Menge wird IN Form einer IDRecordCollection zurckgeliefert.
   Siehe dazu auch Kapitel 5 "Lesen auf der Exemplarebene".
*/
FUNCTION findString(a NUMBER, s VARCHAR2, d TIMESTAMP) RETURN IDRecordCollection IS
	result IDRecordCollection;
BEGIN
	OPEN result FOR 
	SELECT DISTINCT linkValue1.owner,o2.instance_of FROM (SELECT * FROM STRING_VALUE WHERE VALUE LIKE s) stringValue1
		INNER JOIN (SELECT * FROM Link WHERE INSTANCE_OF=a) linkValue1 ON linkValue1.target=stringValue1.ID
		INNER JOIN USER_OBJECT userObject1 ON linkValue1.owner=userObject1.ID
		INNER JOIN SHORT_TRANSACTION st1 ON st1.id=linkValue1.created
		INNER JOIN 
		(SELECT * FROM LONG_TRANSACTION 
			WHERE rightDateGreaterEqualsLeftDate(PUBLISHED, d)=1 ) longTransAction1 ON longTransAction1.ID=st1.container
		INNER JOIN object o2 ON o2.id =linkValue1.owner
	WHERE linkValue1.ID NOT IN (
		SELECT ac1.V FROM ACCESSED ac1
			INNER JOIN SHORT_TRANSACTION st2 ON ac1.T=st2.ID
			INNER JOIN LONG_TRANSACTION lt2 ON lt2.ID=st2.CONTAINER 
		WHERE (ac1.TYPE='D' AND rightDateGreaterEqualsLeftDate(lt2.PUBLISHED, d)=1));
	RETURN result;
END;

/* findInteger(a NUMBER, i NUMBER, d TIMESTAMP) RETURN IDRecordCollection
   Diese Operation realisiert die Suche nach den User_Object-Identifikatoren, welche am Datum <d> einen Link vom Typ <a> auf ein Integer mit dem Wert <i> haben.
   Die entsprechende Menge wird IN Form einer IDRecordCollection zurckgeliefert.
   Siehe dazu auch Kapitel 5 "Lesen auf der Exemplarebene".
*/  
FUNCTION findInteger(a NUMBER, i NUMBER, d TIMESTAMP) RETURN IDRecordCollection IS
	result IDRecordCollection;
BEGIN
	OPEN RESULT FOR
	SELECT DISTINCT linkValue1.owner, o2.instance_of FROM (SELECT * FROM INTEGER_VALUE WHERE VALUE LIKE i) integerValue1
		INNER JOIN (SELECT * FROM Link WHERE INSTANCE_OF=a) linkValue1 ON linkValue1.target=integerValue1.ID
		INNER JOIN USER_OBJECT userObject1 ON linkValue1.owner=userObject1.ID
		INNER JOIN SHORT_TRANSACTION st1 ON st1.id=linkValue1.created
		INNER JOIN 
			(SELECT * FROM LONG_TRANSACTION
				WHERE rightDateGreaterEqualsLeftDate(PUBLISHED, d)=1 ) longTransAction1 ON longTransAction1.ID=st1.container
		INNER JOIN object o2 ON o2.id=linkValue1.owner
	WHERE linkValue1.ID NOT IN 
	(SELECT ac1.V FROM ACCESSED ac1
		INNER JOIN SHORT_TRANSACTION st2 ON ac1.T=st2.ID
		INNER JOIN LONG_TRANSACTION lt2 ON lt2.ID=st2.CONTAINER
	WHERE (ac1.TYPE='D' AND rightDateGreaterEqualsLeftDate(lt2.PUBLISHED, d)=1));
	RETURN result;
END;

/* findStringTransaction(a NUMBER, s VARCHAR2, trans NUMBER) RETURN IDRecordCollection
   Diese Operation realisiert die Suche nach den User_Object-Identifikatoren, welche innerhalb des Sichbereiches der Transaktion <trans> einen Link vom Typ <a> auf einen String mit dem Wert <s> haben.
   Die entsprechende Menge wird IN Form einer IDRecordCollection zurckgeliefert.
   Ist die Transaktion <trans> bereits geschlossen, so wird eine exception geliefert.
   Siehe dazu auch Kapitel 5 "Lesen und Schreiben IN Transaktionen".
*/    
FUNCTION findStringTransaction(a NUMBER, s VARCHAR2, trans NUMBER) RETURN IDRecordCollection IS
	result IDRecordCollection;
	currentID NUMBER;
	instanceID NUMBER;
	accessEntryCoursor IDRecordCollection;
BEGIN
	IF (isTransactionOpen(trans)=0) THEN
		RAISE userexception.not_open_exception;
	END IF;
	accessEntryCoursor	:= queryFindStringByTrans(a ,s ,trans);
	result				:= queryFindStringByTrans(a ,s ,trans);
	LOOP
		FETCH accessEntryCoursor INTO currentID, instanceID;
		EXIT WHEN accessEntryCoursor%NOTFOUND;
		insertReadAccessed(currentID, trans);
	END LOOP;
	CLOSE accessEntryCoursor;
	RETURN result;
END;

/* findIntegerTransaction(a NUMBER, i NUMBER, trans NUMBER) RETURN IDRecordCollection
   Diese Operation realisiert die Suche nach den User_Object-Identifikatoren, welche innerhalb des Sichbereiches der Transaktion <trans> einen Link vom Typ <a> auf ein Integer mit dem Wert <i> haben.
   Die entsprechende Menge wird IN Form einer IDRecordCollection zurckgeliefert.
   Ist die Transaktion <trans> bereits geschlossen, so wird eine exception geliefert.
   Siehe dazu auch Kapitel 5 "Lesen und Schreiben IN Transaktionen".
*/    
FUNCTION findIntegerTransaction(a NUMBER, i NUMBER, trans NUMBER) RETURN IDRecordCollection IS
	result IDRecordCollection;
	currentID NUMBER;
	instanceID NUMBER;
	accessEntryCoursor IDRecordCollection;
BEGIN
	IF (isTransactionOpen(trans)=0) then
		raise userexception.not_open_exception;
	end if;

	accessEntryCoursor := queryFindIntegerByTrans(a, i, trans);
	result             := queryFindIntegerByTrans(a, i, trans);

	LOOP
		FETCH accessEntryCoursor INTO currentID, instanceID;
		EXIT WHEN accessEntryCoursor%NOTFOUND;
		insertReadAccessed(currentID, trans);
		END LOOP;
	close accessEntryCoursor;

	RETURN result;
END;

/* 
  TODO: setObjectWithKey und setObject zusammenfhren
  setObjectWithKey(o NUMBER, a NUMBER, t NUMBER, k NUMBER, lt NUMBER) RETURN NUMBER
  Diese Operation erzeugt einen neuen Link vom Typ <a> mit dem owner <o> und dem target <t> aus der Tabelle Object. <lt> identifiziert den aktuellen Transaktionskontext.
  Liefert die ID des neu erzeugten Links im Fehlerfall gibt es eine entsprechende Ausnahme.
  Ausnahmen treten auf, falls die owner und target ID nicht konform mit der Typisierung der Assoziatione sind. 
  Siehe Dokumentation Kapitel 5 "Lesen und Schreiben IN Transaktionen".
*/
FUNCTION setObjectWithKey(o NUMBER, a NUMBER, t NUMBER, k NUMBER, lt NUMBER) RETURN NUMBER IS
	theSeqIdForLink NUMBER;
BEGIN  

	IF (isUOInScope(o, lt)=FALSE) THEN
		raise userexception.owner_not_in_scope_exception;
	END IF;
	
	IF (isUserObject(t) AND isUOInScope(t, lt)=FALSE) THEN
		raise userexception.owner_not_in_scope_exception;
	END IF;

	/* Konsistenzpruefung definiert auf Seite 13 (ii) AND (iii).*/  
	IF(isSpecializationOfOwner(a,o)=FALSE OR isSpecializationOfTarget(a,t)=FALSE) THEN
		raise userexception.no_valid_link_exception;
	END IF;
	
	IF isUnique(a)=TRUE THEN
		IF getLinkCountOfAssoForOwnerK(lt, o, k, a) != 0 THEN
			unset(getUniqueLinkIdOfAssoForOwnerK(lt, o, k, a),lt);
		END IF;
	END IF;

	IF isEssential(a) THEN
		insertModifiedAccessed(o,lt);
	END IF;

	theSeqIdForLink := InstanceItem_Seq.NEXTVAL;
	INSERT INTO Link(ID,OWNER,TARGET,INSTANCE_OF,CREATED) VALUES(theSeqIdForLink,o,t,a,getActiveShortTransaction(lt));
	RETURN theSeqIdForLink;
END;

/* 
  setObject(o NUMBER, a NUMBER, t NUMBER, lt NUMBER) RETURN NUMBER
  Diese Operation erzeugt einen neuen Link vom Typ <a> mit dem owner <o> und dem target <t> aus der Tabelle Object. <lt> identifiziert den aktuellen Transaktionskontext.
  Liefert die ID des neu erzeugten Links im Fehlerfall gibt es eine entsprechende Ausnahme.
  Ausnahmen treten auf, falls die owner und target ID nicht konform mit der Typisierung der Assoziatione sind. 
  Siehe Dokumentation Kapitel 5 "Lesen und Schreiben IN Transaktionen".
*/
FUNCTION setObject(o NUMBER, a NUMBER, t NUMBER, lt NUMBER) RETURN NUMBER IS
	theSeqIdForLink NUMBER;
BEGIN  

	IF (isUOInScope(o, lt)=FALSE) THEN
		raise userexception.owner_not_in_scope_exception;
	END IF;
	
	IF (isUserObject(t) AND isUOInScope(t, lt)=FALSE) THEN
		raise userexception.owner_not_in_scope_exception;
	END IF;

	/* Konsistenzpruefung definiert auf Seite 13 (ii) AND (iii).*/  
	IF(isSpecializationOfOwner(a,o)=FALSE OR isSpecializationOfTarget(a,t)=FALSE) THEN
		raise userexception.no_valid_link_exception;
	END IF;
	
	IF isUnique(a)=TRUE THEN
		IF getLinkCountOfAssoForOwner(lt, o, a) != 0 THEN
			unset(getUniqueLinkIdOfAssoForOwner(lt, o, a),lt);
		END IF;
	END IF;

	IF isEssential(a) THEN
		insertModifiedAccessed(o,lt);
	END IF;

	theSeqIdForLink := InstanceItem_Seq.NEXTVAL;
	INSERT INTO Link(ID,OWNER,TARGET,INSTANCE_OF,CREATED) VALUES(theSeqIdForLink,o,t,a,getActiveShortTransaction(lt));
	RETURN theSeqIdForLink;
END;

/* 
  setUO(o NUMBER, a NUMBER, tParam NUMBER, lt NUMBER) RETURN NUMBER
  Diese Operation erzeugt einen neuen Link vom Typ <a> mit dem owner <o> und dem target <tParam> aus der Tabelle User_Object. <lt> identifiziert den aktuellen Transaktionskontext.
  Liefert die ID des neu erzeugten Links im Fehlerfall gibt es eine entsprechende Ausnahme.
  Ausnahmen treten auf, falls die owner oder target ID nicht im Kontext der Transaktion sichtbar ist oder owner und target ID nicht konform mit der Typisierung der Assoziatione sind. 
  Siehe Dokumentation Kapitel 5 "Lesen und Schreiben IN Transaktionen".
*/
FUNCTION setUO(o NUMBER, a NUMBER, tParam NUMBER, lt NUMBER) RETURN NUMBER IS
BEGIN 

	IF (isMapAssociation(a)=TRUE) THEN
		raise userexception.no_valid_association_exception;
	END IF;
 
	RETURN setObject(o, a, tParam, lt);
END;

/* 
  setStr(o NUMBER, a NUMBER, stringValue VARCHAR2, lt NUMBER) RETURN NUMBER
  Diese Operation erzeugt einen neuen Link vom Typ <a> mit dem owner <o> aus der Tabelle User_Object und dem target <stringValue>. 
  Existiert kein String mit dem Wert <stringValue>, so wird dieser erzeugt. <lt> identifiziert den aktuellen Transaktionskontext.
  Liefert die ID des neu erzeugten Links, im Fehlerfall gibt es eine entsprechende Ausnahme.
  Ausnahmen treten auf, falls der owner nicht im Kontext der Transaktion sichtbar ist oder owner und target nicht konform mit der Typisierung der Assoziation sind. 
  Siehe Dokumentation Kapitel 5 "Lesen und Schreiben IN Transaktionen".
*/
FUNCTION setStr(o NUMBER, a NUMBER, stringValue VARCHAR2, lt NUMBER) RETURN NUMBER IS
	tParam NUMBER;
BEGIN
	tParam := objectfacade.getStringObject(stringValue);
	
	IF (isMapAssociation(a)=TRUE) THEN
		raise userexception.no_valid_association_exception;
	END IF;
	
	RETURN setObject(o, a, tParam, lt);
END;

/* 
  setInt(o NUMBER, a NUMBER, intValue NUMBER, lt NUMBER) RETURN NUMBER
  Diese Operation erzeugt einen neuen Link vom Typ <a> mit dem owner <o> aus der Tabelle User_Object und dem target <intValue>. 
  Existiert kein Integer mit dem Wert <intValue>, so wird dieser erzeugt. <lt> identifiziert den aktuellen Transaktionskontext.
  Liefert die ID des neu erzeugten Links, im Fehlerfall gibt es eine entsprechende Ausnahme.
  Ausnahmen treten auf, falls der owner nicht im Kontext der Transaktion sichtbar ist oder owner und target nicht konform mit der Typisierung der Assoziatione sind. 
  Siehe Dokumentation Kapitel 5 "Lesen und Schreiben IN Transaktionen".
*/
FUNCTION setInt(o NUMBER, a NUMBER, intValue NUMBER, lt NUMBER) RETURN NUMBER IS
	tParam NUMBER;
BEGIN
	tParam := objectfacade.getIntegerObject(intValue);
	
	IF (isMapAssociation(a)=TRUE) THEN
		raise userexception.no_valid_association_exception;
	END IF;
	
	RETURN setObject(o, a, tParam, lt);
END;


/* 
  putObject(o NUMBER, a NUMBER, t NUMBER, k NUMBER, trans NUMBER) RETURN NUMBER
  Diese Operation erzeugt einen neuen Link vom Typ <a> mit dem owner <o> aus der Tabelle User_Object, dem target <t> und dem Key <k>. 
  <trans> identifiziert den aktuellen Transaktionskontext.
  Liefert die ID des neu erzeugten Links, im Fehlerfall gibt es eine entsprechende Ausnahme.
  Welche Ausnahmen auftreten knnen entnehmen Sie bitte der Dokumentation ab Kapitel 5
*/
FUNCTION putObject(o NUMBER, a NUMBER, t NUMBER, k NUMBER, trans NUMBER) RETURN NUMBER IS 
	link NUMBER;
BEGIN
	IF (isMapAssociation(a)=FALSE) THEN
		raise userexception.no_valid_association_exception;
	END IF;

	IF (isUserObject(k) AND isUOInScope(k, trans)=FALSE) THEN
		raise userexception.owner_not_in_scope_exception;
	END IF;	
	
	IF(isSpecializationOfKey(a,k)=FALSE) THEN
		raise userexception.no_valid_link_exception;
	END IF;
	
	IF (isUnique(a) AND getLinkCountOfAssoForOwnerK(trans, o, k, a)!=0) THEN
		raise userexception.duplicate_entry_exception;
	END IF;
	
	link := setObjectWithKey(o, a, t, k, trans);
	insert INTO MapLink(ID, map_KEY) VALUES (link, k);
	RETURN link;
END;

/* 
  putUOUO(o NUMBER, a NUMBER, t NUMBER, k NUMBER, trans NUMBER) RETURN NUMBER
  Diese Operation erzeugt einen neuen Link vom Typ <a> mit dem owner <o> aus der Tabelle User_Object, dem target <t> (UserObject) und dem Key <k> (UserObject). 
  <trans> identifiziert den aktuellen Transaktionskontext.
  Liefert die ID des neu erzeugten Links, im Fehlerfall gibt es eine entsprechende Ausnahme.
  Welche Ausnahmen auftreten knnen entnehmen Sie bitte der Dokumentation ab Kapitel 5
*/
FUNCTION putUOUO(o NUMBER, a NUMBER, t NUMBER, k NUMBER, trans NUMBER) RETURN NUMBER IS
BEGIN
	RETURN putObject(o, a, t, k, trans);
END;

/* 
  putUOStr(o NUMBER, a NUMBER, t NUMBER, k VARCHAR2, trans NUMBER) RETURN NUMBER
  Diese Operation erzeugt einen neuen Link vom Typ <a> mit dem owner <o> aus der Tabelle User_Object, dem target <t> (UserObject) und dem Key <k> (String). 
  <trans> identifiziert den aktuellen Transaktionskontext.
  Liefert die ID des neu erzeugten Links, im Fehlerfall gibt es eine entsprechende Ausnahme.
  Welche Ausnahmen auftreten knnen entnehmen Sie bitte der Dokumentation ab Kapitel 5
*/
FUNCTION putUOStr(o NUMBER, a NUMBER, t NUMBER, k VARCHAR2, trans NUMBER) RETURN NUMBER IS
	kobj NUMBER;
BEGIN
	kobj := objectfacade.getStringObject(k);
	
	RETURN putObject(o, a, t, kobj, trans);
END;

/* 
  putUOInt(o NUMBER, a NUMBER, t NUMBER, k NUMBER, trans NUMBER) RETURN NUMBER
  Diese Operation erzeugt einen neuen Link vom Typ <a> mit dem owner <o> aus der Tabelle User_Object, dem target <t> (UserObject) und dem Key <k> (Integer). 
  <trans> identifiziert den aktuellen Transaktionskontext.
  Liefert die ID des neu erzeugten Links, im Fehlerfall gibt es eine entsprechende Ausnahme.
  Welche Ausnahmen auftreten knnen entnehmen Sie bitte der Dokumentation ab Kapitel 5
*/
FUNCTION putUOInt(o NUMBER, a NUMBER, t NUMBER, k NUMBER, trans NUMBER) RETURN NUMBER IS
	kobj NUMBER;
BEGIN
	kobj := objectfacade.getIntegerObject(k);
	
	RETURN putObject(o, a, t, kobj, trans);
END;

/* 
  putStrUO(o NUMBER, a NUMBER, t VARCHAR2, k NUMBER, trans NUMBER) RETURN NUMBER
  Diese Operation erzeugt einen neuen Link vom Typ <a> mit dem owner <o> aus der Tabelle User_Object, dem target <t> (String) und dem Key <k> (UserObject). 
  <trans> identifiziert den aktuellen Transaktionskontext.
  Liefert die ID des neu erzeugten Links, im Fehlerfall gibt es eine entsprechende Ausnahme.
  Welche Ausnahmen auftreten knnen entnehmen Sie bitte der Dokumentation ab Kapitel 5
*/
FUNCTION putStrUO(o NUMBER, a NUMBER, t VARCHAR2, k NUMBER, trans NUMBER) RETURN NUMBER IS
	tobj NUMBER;
BEGIN
	tobj := objectfacade.getStringObject(t);
	
	RETURN putObject(o, a, tobj, k, trans);
END;

/* 
  putStrStr(o NUMBER, a NUMBER, t VARCHAR2, k VARCHAR2, trans NUMBER) RETURN NUMBER
  Diese Operation erzeugt einen neuen Link vom Typ <a> mit dem owner <o> aus der Tabelle User_Object, dem target <t> (String) und dem Key <k> (String). 
  <trans> identifiziert den aktuellen Transaktionskontext.
  Liefert die ID des neu erzeugten Links, im Fehlerfall gibt es eine entsprechende Ausnahme.
  Welche Ausnahmen auftreten knnen entnehmen Sie bitte der Dokumentation ab Kapitel 5
*/
FUNCTION putStrStr(o NUMBER, a NUMBER, t VARCHAR2, k VARCHAR2, trans NUMBER) RETURN NUMBER IS
	tobj NUMBER;
	kobj NUMBER;
BEGIN
	tobj := objectfacade.getStringObject(t);
	kobj := objectfacade.getStringObject(k);
	
	RETURN putObject(o, a, tobj, kobj, trans);
END;

/* 
  putStrInt(o NUMBER, a NUMBER, t VARCHAR2, k NUMBER, trans NUMBER) RETURN NUMBER
  Diese Operation erzeugt einen neuen Link vom Typ <a> mit dem owner <o> aus der Tabelle User_Object, dem target <t> (String) und dem Key <k> (Integer). 
  <trans> identifiziert den aktuellen Transaktionskontext.
  Liefert die ID des neu erzeugten Links, im Fehlerfall gibt es eine entsprechende Ausnahme.
  Welche Ausnahmen auftreten knnen entnehmen Sie bitte der Dokumentation ab Kapitel 5
*/
FUNCTION putStrInt(o NUMBER, a NUMBER, t VARCHAR2, k NUMBER, trans NUMBER) RETURN NUMBER IS
	tobj NUMBER;
	kobj NUMBER;
BEGIN
	tobj := objectfacade.getStringObject(t);
	kobj := objectfacade.getIntegerObject(k);
	
	RETURN putObject(o, a, tobj, kobj, trans);
END;

/* 
  putIntUO(o NUMBER, a NUMBER, t NUMBER, k NUMBER, trans NUMBER) RETURN NUMBER
  Diese Operation erzeugt einen neuen Link vom Typ <a> mit dem owner <o> aus der Tabelle User_Object, dem target <t> (Integer) und dem Key <k> (UserObject). 
  <trans> identifiziert den aktuellen Transaktionskontext.
  Liefert die ID des neu erzeugten Links, im Fehlerfall gibt es eine entsprechende Ausnahme.
  Welche Ausnahmen auftreten knnen entnehmen Sie bitte der Dokumentation ab Kapitel 5
*/
FUNCTION putIntUO(o NUMBER, a NUMBER, t NUMBER, k NUMBER, trans NUMBER) RETURN NUMBER IS
	tobj NUMBER;
BEGIN
	tobj := objectfacade.getIntegerObject(t);
	
	RETURN putObject(o, a, tobj, k, trans);
END;

/* 
  putIntStr(o NUMBER, a NUMBER, t NUMBER, k VARCHAR2, trans NUMBER) RETURN NUMBER
  Diese Operation erzeugt einen neuen Link vom Typ <a> mit dem owner <o> aus der Tabelle User_Object, dem target <t> (Integer) und dem Key <k> (String). 
  <trans> identifiziert den aktuellen Transaktionskontext.
  Liefert die ID des neu erzeugten Links, im Fehlerfall gibt es eine entsprechende Ausnahme.
  Welche Ausnahmen auftreten knnen entnehmen Sie bitte der Dokumentation ab Kapitel 5
*/
FUNCTION putIntStr(o NUMBER, a NUMBER, t NUMBER, k VARCHAR2, trans NUMBER) RETURN NUMBER IS
	tobj NUMBER;
	kobj NUMBER;
BEGIN
	tobj := objectfacade.getIntegerObject(t);
	kobj := objectfacade.getStringObject(k);
	
	RETURN putObject(o, a, tobj, kobj, trans);
END;

/* 
  putIntInt(o NUMBER, a NUMBER, t NUMBER, k NUMBER, trans NUMBER) RETURN NUMBER
  Diese Operation erzeugt einen neuen Link vom Typ <a> mit dem owner <o> aus der Tabelle User_Object, dem target <t> (Integer) und dem Key <k> (Integer). 
  <trans> identifiziert den aktuellen Transaktionskontext.
  Liefert die ID des neu erzeugten Links, im Fehlerfall gibt es eine entsprechende Ausnahme.
  Welche Ausnahmen auftreten knnen entnehmen Sie bitte der Dokumentation ab Kapitel 5
*/
FUNCTION putIntInt(o NUMBER, a NUMBER, t NUMBER, k NUMBER, trans NUMBER) RETURN NUMBER IS
	tobj NUMBER;
	kobj NUMBER;
BEGIN
	tobj := objectfacade.getIntegerObject(t);
	kobj := objectfacade.getIntegerObject(k);
	
	RETURN putObject(o, a, tobj, kobj, trans);
END;

/*
	Get(o NUMBER, a NUMBER, d TIMESTAMP) RETURN LinkObjectRecordCollection 
	Die Funktion Get liefert alle Links und ihre Target-Objects, die das UserObject mit der Id <o> als Owner haben,
	von der Assoziation <a> sind und zu dem Zeitpunkt <d> erstellt und nicht gelscht sind.
*/  
FUNCTION get(o NUMBER, a NUMBER, d TIMESTAMP) RETURN LinkObjectRecordCollection IS
	RESULT LinkObjectRecordCollection;
BEGIN OPEN RESULT FOR
	SELECT DISTINCT Link.id, Link.instance_of, object.id,object.instance_of FROM user_object
		INNER JOIN Link ON Link.owner=user_object.id
		INNER JOIN Association ON Association.id=Link.instance_of
		INNER JOIN object ON object.id=Link.target 
		INNER JOIN short_transaction st1 ON st1.id=Link.created
		INNER JOIN long_transaction lt1 ON lt1.id=st1.container
	WHERE user_object.id=o
		AND Association.id=a
		AND lt1.published <= d
		AND 
		user_object.id NOT IN
			(SELECT user_object.id FROM accessed
				INNER JOIN short_transaction st2 ON st2.id=accessed.t
				INNER JOIN long_transaction lt2 ON lt2.id=st2.container
			WHERE accessed.v=user_object.id
				AND accessed.TYPE='D'
				AND lt2.published <= d);
	RETURN RESULT;
END get;


/*
	GetMap(o NUMBER, a NUMBER, k NUMBER, d TIMESTAMP) RETURN UserObjectRecordCollection 
	Die Funktion GetMap liefert alle Links und ihre Target-Objects, die das UserObject mit der Id <o> als Owner und den map-key der Id <key> haben,
	von der Assoziation <a> sind und zu dem Zeitpunkt <d> erstellt und nicht gelscht sind.
*/  
FUNCTION getMap(o NUMBER, a NUMBER, k NUMBER, d TIMESTAMP) RETURN UserObjectRecordCollection IS
	RESULT UserObjectRecordCollection;
BEGIN OPEN RESULT FOR
	SELECT DISTINCT object.id,object.instance_of FROM user_object
		INNER JOIN Link ON Link.owner=user_object.id
		INNER JOIN MapLink ON MapLink.id=Link.id
		INNER JOIN Association ON Association.id=Link.instance_of
		INNER JOIN object ON object.id=Link.target 
		INNER JOIN short_transaction st1 ON st1.id=Link.created
		INNER JOIN long_transaction lt1 ON lt1.id=st1.container
	WHERE user_object.id=o
		AND Association.id=a
		AND lt1.published <= d
		AND MapLink.map_key=k
		AND 
		user_object.id NOT IN
			(SELECT user_object.id FROM accessed
				INNER JOIN short_transaction st2 ON st2.id=accessed.t
				INNER JOIN long_transaction lt2 ON lt2.id=st2.container
			WHERE accessed.v=user_object.id
				AND accessed.TYPE='D'
				AND lt2.published <= d);
	RETURN RESULT;
END;

/*
	getMapUO(o NUMBER, a NUMBER, k NUMBER, d TIMESTAMP) RETURN UserObjectRecordCollection 
	Die Funktion GetMapUO liefert alle Links und ihre Target-Objects, die das UserObject mit der Id <o> als Owner und den map-key der Id <k> haben,
	von der Assoziation <a> sind und zu dem Zeitpunkt <d> erstellt und nicht gelscht sind.
*/  
FUNCTION getMapUO(o NUMBER, a NUMBER, k NUMBER, d TIMESTAMP) RETURN UserObjectRecordCollection IS
BEGIN 
	RETURN getMap(o, a, k, d);
END;

/*
	getMapStr(o NUMBER, a NUMBER, k VARCHAR2, d TIMESTAMP) RETURN UserObjectRecordCollection 
	Die Funktion GetMapStr liefert alle Links und ihre Target-Objects, die das UserObject mit der Id <o> als Owner und den map-key als String <k> haben,
	von der Assoziation <a> sind und zu dem Zeitpunkt <d> erstellt und nicht gelscht sind.
*/  
FUNCTION getMapStr(o NUMBER, a NUMBER, k VARCHAR2, d TIMESTAMP) RETURN UserObjectRecordCollection IS
	kobj NUMBER;
BEGIN 
	kobj := objectfacade.getStringObject(k);
	RETURN getMap(o, a, kobj, d);
END;

/*
	getMapInt(o NUMBER, a NUMBER, k NUMBER, d TIMESTAMP) RETURN UserObjectRecordCollection 
	Die Funktion GetMapInt liefert alle Links und ihre Target-Objects, die das UserObject mit der Id <o> als Owner und den map-key als Int <k> haben,
	von der Assoziation <a> sind und zu dem Zeitpunkt <d> erstellt und nicht gelscht sind.
*/  
FUNCTION getMapInt(o NUMBER, a NUMBER, k NUMBER, d TIMESTAMP) RETURN UserObjectRecordCollection IS
	kobj NUMBER;
BEGIN 
	kobj := objectfacade.getIntegerObject(k);
	RETURN getMap(o, a, kobj, d);
END;


/*
	inverseGet(o NUMBER, a NUMBER, d TIMESTAMP) RETURN LinkUserObjectRecordCollection	
	Die Funktion inverseGet liefert alle Links und ihre Owner-UserObjects, die das UserObject mit der Id <o> als Target haben,
	von der Assoziation <a> sind und zu dem Zeitpunkt <d> erstellt und nicht gelscht sind.
*/
FUNCTION inverseGet(o NUMBER, a NUMBER, d TIMESTAMP) RETURN LinkUserObjectRecordCollection IS
	RESULT LinkUserObjectRecordCollection;
BEGIN OPEN RESULT FOR
	SELECT DISTINCT Link.id, Link.instance_of, o2.id, o2.instance_of FROM user_object
		INNER JOIN Link ON Link.target=user_object.id
		INNER JOIN Association ON Association.id=Link.instance_of
		INNER JOIN object o2 ON o2.id=Link.owner
		INNER JOIN short_transaction st1 ON st1.id=Link.created
		INNER JOIN long_transaction lt1 ON lt1.id=st1.container
	WHERE user_object.id=o
		AND Association.id=a
		AND lt1.published <= d
		AND 
		o2.id NOT IN
			(SELECT user_object.id FROM accessed
				INNER JOIN short_transaction st2 ON st2.id=accessed.t
				INNER JOIN long_transaction lt2 ON lt2.id=st2.container
			WHERE accessed.v=user_object.id
				AND accessed.TYPE='D'
				AND lt2.published <= d);
	RETURN RESULT;
END inverseGet;


/*
	inverseGetMap(o NUMBER, a NUMBER, k NUMBER, d TIMESTAMP) RETURN ObjectKeyLinkRecordCollection	
	Die Funktion inverseGet liefert alle Links, ihre Owner-UserObjects und Keys, die das UserObject mit der Id <o> als Target haben,
	von der Assoziation <a> sind und zu dem Zeitpunkt <d> erstellt und nicht gelscht sind.
*/
FUNCTION inverseGetMap(o NUMBER, a NUMBER, d TIMESTAMP) RETURN ObjectKeyLinkRecordCollection IS
	RESULT ObjectKeyLinkRecordCollection;
BEGIN OPEN RESULT FOR
	SELECT DISTINCT o2.id, o2.instance_of, MapLink.map_key, MapAssociation.map_key_type, Link.id, Link.instance_of FROM user_object
		INNER JOIN Link ON Link.target=user_object.id
		INNER JOIN MapLink ON MapLink.id=Link.id
		INNER JOIN Association ON Association.id=Link.instance_of
		INNER JOIN MapAssociation ON MapAssociation.id=Link.instance_of
		INNER JOIN object o2 ON o2.id=Link.owner
		INNER JOIN short_transaction st1 ON st1.id=Link.created
		INNER JOIN long_transaction lt1 ON lt1.id=st1.container
	WHERE user_object.id=o
		AND Association.id=a
		AND lt1.published <= d
		AND 
		o2.id NOT IN
			(SELECT user_object.id FROM accessed
				INNER JOIN short_transaction st2 ON st2.id=accessed.t
				INNER JOIN long_transaction lt2 ON lt2.id=st2.container
			WHERE accessed.v=user_object.id
				AND accessed.TYPE='D'
				AND lt2.published <= d);
	RETURN RESULT;
END inverseGetMap;


/*
	inverseGetMapUO(o NUMBER, a NUMBER, d TIMESTAMP) RETURN ObjectKeyLinkRecordCollection	
	Die Funktion inverseGetMapUO liefert alle Links, ihre Owner-UserObjects und die Keys, die das UserObject mit der Id <o> als Target
	von der Assoziation <a> sind und zu dem Zeitpunkt <d> erstellt und nicht gelscht sind.
*/ 
FUNCTION inverseGetMapUO(o NUMBER, a NUMBER, d TIMESTAMP) RETURN ObjectKeyLinkRecordCollection IS
BEGIN 
	RETURN inverseGetMap(o, a, d);
END;


/*
	inverseGetMapStr(o VARCHAR2, a NUMBER, d TIMESTAMP) RETURN ObjectKeyLinkRecordCollection	
	Die Funktion inverseGetMapStr liefert alle Links, ihre Owner-UserObjects und die Keys, die das UserObject mit der Id <o> als Target
	von der Assoziation <a> sind und zu dem Zeitpunkt <d> erstellt und nicht gelscht sind.
*/
FUNCTION inverseGetMapStr(o VARCHAR2, a NUMBER, d TIMESTAMP) RETURN ObjectKeyLinkRecordCollection IS
	tobj NUMBER;
BEGIN 
	tobj := objectfacade.getStringObject(o);
	RETURN inverseGetMap(tobj, a, d);
END;


/*
	inverseGetMapInt(o NUMBER, a NUMBER, d TIMESTAMP) RETURN ObjectKeyLinkRecordCollection
	Die Funktion inverseGetMapInt liefert alle Links, ihre Owner-UserObjects und die Keys, die das UserObject mit der Id <o> als Target
	von der Assoziation <a> sind und zu dem Zeitpunkt <d> erstellt und nicht gelscht sind.
*/
FUNCTION inverseGetMapInt(o NUMBER, a NUMBER, d TIMESTAMP) RETURN ObjectKeyLinkRecordCollection IS
	tobj NUMBER;
BEGIN 
	tobj := objectfacade.getIntegerObject(o);
	RETURN inverseGetMap(tobj, a, d);
END;




/*
	getTransaction(o NUMBER, a NUMBER, trans NUMBER) RETURN LinkObjectRecordCollection
	Die Funktion getTransaction liefert alle Links und ihre Target-Objects, die das UserObject mit der Id <o> als Owner haben,
	von der Assoziation <a> sind und innerhalb der langen Transaktion mit der Id <trans> erstellt und nicht gelscht sind.
	Fr die Links und Target-Objects wird je ein gelesen-Accessed Vermerk eingefgt.
*/
FUNCTION getTransaction(o NUMBER, a NUMBER, trans NUMBER) RETURN LinkObjectRecordCollection IS
	result LinkObjectRecordCollection;
	resultRow LinkObjectRecord;
BEGIN
	result := objectfacade.getTransactionCursor(o,a,trans);
	LOOP
		FETCH result INTO resultRow;
		EXIT WHEN result%NOTFOUND;
		objectfacade.insertReadAccessed(resultRow.objectid, trans);
		objectfacade.insertReadAccessed(resultRow.linkid, trans);
	END LOOP;
	CLOSE result;
	result := objectfacade.getTransactionCursor(o,a,trans);
	RETURN result;
END getTransaction;

/*
	getMapTransaction(o NUMBER, a NUMBER, k NUMBER, trans NUMBER) RETURN UserObjectRecordCollection
	Die Funktion getMapTransaction liefert alle Links und ihre Target-Objects, die das UserObject mit der Id <o> als Owner und den map-key der Id <key> haben,
	von der Assoziation <a> sind und innerhalb der langen Transaktion mit der Id <trans> erstellt und nicht gelscht sind.
	Fr die Links und Target-Objects wird je ein gelesen-Accessed Vermerk eingefgt.
*/
FUNCTION getMapTransaction(o NUMBER, a NUMBER, k NUMBER, trans NUMBER) RETURN ObjectKeyLinkRecordCollection IS
	result ObjectKeyLinkRecordCollection;
	resultRow ObjectKeyLinkRecord;
BEGIN
	result := objectfacade.getMapTransactionCursor(o,a,k,trans);
	LOOP
		FETCH result INTO resultRow;
		EXIT WHEN result%NOTFOUND;
		objectfacade.insertReadAccessed(resultRow.objectid, trans);
		objectfacade.insertReadAccessed(resultRow.linkid, trans);
	END LOOP;
	CLOSE result;
	result := objectfacade.getMapTransactionCursor(o,a,k,trans);
	RETURN result;
END getMapTransaction;

/*
	getMapIntTransaction(o NUMBER, a NUMBER, k NUMBER, trans NUMBER) RETURN ObjectKeyLinkRecordCollection
	Die Funktion getMapIntTransaction liefert alle Links und ihre Target-Objects, die das UserObject mit der Id <o> als Owner und den map-key <k> haben,
	von der Assoziation <a> sind und innerhalb der langen Transaktion mit der Id <trans> erstellt und nicht gelscht sind.
	Fr die Links und Target-Objects wird je ein gelesen-Accessed Vermerk eingefgt.
*/ 
FUNCTION getMapUOTransaction(o NUMBER, a NUMBER, k NUMBER, trans NUMBER) RETURN ObjectKeyLinkRecordCollection IS
BEGIN 
	RETURN getMapTransaction(o, a, k, trans);
END;

/*
	getMapStrTransaction(o NUMBER, a NUMBER, k VARCHAR2, trans NUMBER) RETURN ObjectKeyLinkRecordCollection
	Die Funktion getMapStrTransaction liefert alle Links und ihre Target-Objects, die das UserObject mit der Id <o> als Owner und den map-key <k> haben,
	von der Assoziation <a> sind und innerhalb der langen Transaktion mit der Id <trans> erstellt und nicht gelscht sind.
	Fr die Links und Target-Objects wird je ein gelesen-Accessed Vermerk eingefgt.
*/ 
FUNCTION getMapStrTransaction(o NUMBER, a NUMBER, k VARCHAR2, trans NUMBER) RETURN ObjectKeyLinkRecordCollection IS
	kobj NUMBER;
BEGIN 
	kobj := objectfacade.getStringObject(k);
	RETURN getMapTransaction(o, a, kobj, trans);
END;

/*
	getMapIntTransaction(o NUMBER, a NUMBER, k NUMBER, trans NUMBER) RETURN ObjectKeyLinkRecordCollection
	Die Funktion getMapIntTransaction liefert alle Links und ihre Target-Objects, die das UserObject mit der Id <o> als Owner und den map-key <k> haben,
	von der Assoziation <a> sind und innerhalb der langen Transaktion mit der Id <trans> erstellt und nicht gelscht sind.
	Fr die Links und Target-Objects wird je ein gelesen-Accessed Vermerk eingefgt.
*/ 
FUNCTION getMapIntTransaction(o NUMBER, a NUMBER, k NUMBER, trans NUMBER) RETURN ObjectKeyLinkRecordCollection IS
	kobj NUMBER;
BEGIN 
	kobj := objectfacade.getIntegerObject(k);
	RETURN getMapTransaction(o, a, kobj, trans);
END;



/*
	getTransactionCursor(o NUMBER, a NUMBER, trans NUMBER) RETURN LinkObjectRecordCollection
	Die Hilfsfunktion getTransactionCursor liefert einen offenen Cursor, der alle Links und ihre Target-Objects, die das UserObject mit der Id <o> als Owner haben,
	von der Assoziation <a> sind und innerhalb der langen Transaktion mit der Id <trans> erstellt und nicht gelöscht sind enthält.
*/
FUNCTION getTransactionCursor(o NUMBER, a NUMBER, trans NUMBER) RETURN LinkObjectRecordCollection IS
	result LinkObjectRecordCollection;
BEGIN 
	OPEN result FOR
		SELECT DISTINCT l.id, l.instance_of, object.id, object.instance_of FROM Link l
			INNER JOIN object ON object.id=l.target
			INNER JOIN short_transaction ON short_transaction.id=l.created
			INNER JOIN long_transaction ON long_transaction.id=short_transaction.container
		WHERE l.owner=o
			AND l.instance_of=a
			AND objectfacade.rightdategreaterequalsleftdate(long_transaction.published,sysdate)=1
			AND l.id NOT IN
			(SELECT Link.id FROM Link INNER JOIN accessed ON accessed.v=Link.id
				INNER JOIN short_transaction ON short_transaction.id=accessed.t
				INNER JOIN long_transaction ON long_transaction.id=short_transaction.container
			WHERE accessed.v=l.id
				AND accessed.type='D'
				AND 
				(objectfacade.rightdategreaterequalsleftdate(long_transaction.published,sysdate)=1 OR long_transaction.id=trans))
	UNION
		SELECT DISTINCT Link.id, Link.instance_of, object.id, object.instance_of FROM Link
			INNER JOIN object ON object.id=Link.target
			INNER JOIN short_transaction ON short_transaction.id=Link.created
		WHERE Link.owner=o
			AND Link.instance_of=a
			AND short_transaction.container=trans;   
	RETURN result;
END getTransactionCursor;

/*
	getMapTransactionCursor(o NUMBER, a NUMBER, k NUMBER, trans NUMBER) RETURN UserObjectRecordCollection
	Die Hilfsfunktion getMapTransactionCursor liefert einen offenen Cursor, der alle Links und ihre Target-Objects, die das UserObject mit der Id <o> als Owner und den map-key der Id <key> haben,
	von der Assoziation <a> sind und innerhalb der langen Transaktion mit der Id <trans> erstellt und nicht gelscht sind enthlt.
*/
FUNCTION getMapTransactionCursor(o NUMBER, a NUMBER, k NUMBER, trans NUMBER) RETURN ObjectKeyLinkRecordCollection IS
	result ObjectKeyLinkRecordCollection;
BEGIN OPEN result FOR
	SELECT DISTINCT object.id, object.instance_of, MapLink.map_key, MapAssociation.map_key_type, l.id, l.instance_of FROM Link l
		INNER JOIN MapLink ON MapLink.id=l.id
		INNER JOIN object ON object.id=l.target
		INNER JOIN short_transaction ON short_transaction.id=l.created
		INNER JOIN long_transaction ON long_transaction.id=short_transaction.container
		INNER JOIN MapAssociation ON MapAssociation.id=l.instance_of
	WHERE l.owner=o
		AND l.instance_of=a
		AND MapLink.map_key=k
		AND objectfacade.rightdategreaterequalsleftdate(long_transaction.published,sysdate)=1
		AND l.id NOT IN
		(SELECT Link.id FROM Link INNER JOIN accessed ON accessed.v=Link.id
			INNER JOIN short_transaction ON short_transaction.id=accessed.t
			INNER JOIN long_transaction ON long_transaction.id=short_transaction.container
		WHERE accessed.v=l.id
			AND accessed.type='D'
			AND
			(objectfacade.rightdategreaterequalsleftdate(long_transaction.published,sysdate)=1 OR long_transaction.id=trans))
	UNION
		SELECT DISTINCT object.id, object.instance_of, ltd.map_key, MapAssociation.map_key_type, Link.id, Link.instance_of FROM Link
			INNER JOIN MapLink ltd ON ltd.id=Link.id
			INNER JOIN object ON object.id=Link.target
			INNER JOIN short_transaction ON short_transaction.id=Link.created
			INNER JOIN MapAssociation ON MapAssociation.id=Link.instance_of
		WHERE Link.owner=o
			AND Link.instance_of=a
			AND ltd.map_key=k
			AND short_transaction.container=trans;   
	RETURN result;
END getMapTransactionCursor;

/*
	inverseGetMapTransaction(o NUMBER, a NUMBER, k NUMBER, trans NUMBER) RETURN ObjectKeyLinkRecordCollection
	Die Funktion inverseGetMapTransaction liefert alle Links, ihre Owner-UserObjects und Keys, die das UserObject mit der Id <o> als Target,
	von der Assoziation <a> sind und innerhalb der langen Transaktion mit der Id <trans> erstellt und nicht gelscht sind.
	Fr die Links, Owner-Objects und Keys wird je ein gelesen-Accessed Vermerk eingefgt.
*/
FUNCTION inverseGetMapTransaction(o NUMBER, a NUMBER, trans NUMBER) RETURN ObjectKeyLinkRecordCollection IS 
	result ObjectKeyLinkRecordCollection;
	resultRow ObjectKeyLinkRecord;
BEGIN
	result := objectfacade.inverseGetMapTransactionCursor(o,a,trans);
	LOOP
		FETCH result INTO resultRow;
		EXIT WHEN result%NOTFOUND;
		objectfacade.insertReadAccessed(resultRow.objectid, trans); 
		objectfacade.insertReadAccessed(resultRow.linkid, trans);
		objectfacade.insertReadAccessed(resultRow.keyid, trans);
	END LOOP;
	CLOSE result;
	result := objectfacade.inverseGetMapTransactionCursor(o,a,trans);
	RETURN RESULT;
END inverseGetMapTransaction;

/*
	inverseGetMapUOTransaction(o NUMBER, a NUMBER, trans NUMBER) RETURN ObjectKeyLinkRecordCollection
	Die Funktion inverseGetMapTransaction liefert alle Links, ihre Owner-UserObjects und ihre Keys, die das UserObject mit der Id <o> als Target,
	von der Assoziation <a> sind und innerhalb der langen Transaktion mit der Id <trans> erstellt und nicht gelscht sind.
	Fr die Links, Owner-Objects und Keys-Onjekte wird je ein gelesen-Accessed Vermerk eingefgt.
*/
FUNCTION inverseGetMapUOTransaction(o NUMBER, a NUMBER, trans NUMBER) RETURN ObjectKeyLinkRecordCollection IS
BEGIN 
	RETURN inverseGetMapTransaction(o, a, trans);
END;

/*
	inverseGetMapStrTransaction(o VARCHAR2, a NUMBER, trans NUMBER) RETURN ObjectKeyLinkRecordCollection
	Die Funktion inverseGetMapTransaction liefert alle Links, ihre Owner-UserObjects und ihre Keys, die das UserObject mit der Id <o> als Target,
	von der Assoziation <a> sind und innerhalb der langen Transaktion mit der Id <trans> erstellt und nicht gelscht sind.
	Fr die Links, Owner-Objects und Keys-Onjekte wird je ein gelesen-Accessed Vermerk eingefgt.
*/
FUNCTION inverseGetMapStrTransaction(o VARCHAR2, a NUMBER, trans NUMBER) RETURN ObjectKeyLinkRecordCollection IS
	tobj NUMBER;
BEGIN 
	tobj := objectfacade.getStringObject(o);
	RETURN inverseGetMapTransaction(tobj, a, trans);
END;
/*
	inverseGetMapIntTransaction(o NUMBER, a NUMBER, trans NUMBER) RETURN ObjectKeyLinkRecordCollection
	Die Funktion inverseGetMapTransaction liefert alle Links, ihre Owner-UserObjects und ihre Keys, die das UserObject mit der Id <o> als Target,
	von der Assoziation <a> sind und innerhalb der langen Transaktion mit der Id <trans> erstellt und nicht gelscht sind.
	Fr die Links, Owner-Objects und Keys-Onjekte wird je ein gelesen-Accessed Vermerk eingefgt.
*/
FUNCTION inverseGetMapIntTransaction(o NUMBER, a NUMBER, trans NUMBER) RETURN ObjectKeyLinkRecordCollection IS
	tobj NUMBER;
BEGIN 
	tobj := objectfacade.getIntegerObject(o);
	RETURN inverseGetMapTransaction(tobj, a, trans);
END;
/*
	inverseGetTransaction(o NUMBER, a NUMBER, trans NUMBER) RETURN LinkUserObjectRecordCollection
	Die Funktion inverseGetTransaction liefert alle Links und ihre Owner-UserObjects, die das UserObject mit der Id <o> als Target haben,
	von der Assoziation <a> sind und innerhalb der langen Transaktion mit der Id <trans> erstellt und nicht gelscht sind.
	Fr die Links und Owner-Objects wird je ein gelesen-Accessed Vermerk eingefgt.
*/
FUNCTION inverseGetTransaction(o NUMBER, a NUMBER, trans NUMBER) RETURN LinkUserObjectRecordCollection IS 
	result LinkUserObjectRecordCollection;
	resultRow LinkUserObjectRecord;
BEGIN
	result := objectfacade.inverseGetTransactionCursor(o,a,trans);
	LOOP
		FETCH result INTO resultRow;
		EXIT WHEN result%NOTFOUND;
		objectfacade.insertReadAccessed(resultRow.userobjectid, trans); 
		objectfacade.insertReadAccessed(resultRow.linkid, trans); 
	END LOOP;
	CLOSE result;
	result := objectfacade.inverseGetTransactionCursor(o,a,trans);
	RETURN RESULT;
END inverseGetTransaction;

/*
	inverseGetTransactionCursor(o NUMBER, a NUMBER, trans NUMBER) RETURN LinkUserObjectRecordCollection
	Die Hilfsfunktion inverseGetTransactionCursor liefert einen offenen Cursor der, alle Links und ihre Owner-UserObjects, die das UserObject mit der Id <o> als Target haben,
	von der Assoziation <a> sind und innerhalb der langen Transaktion mit der Id <trans> erstellt und nicht gelscht sind enthlt.
*/
FUNCTION inverseGetTransactionCursor(o NUMBER, a NUMBER, trans NUMBER) RETURN LinkUserObjectRecordCollection IS 
	result LinkUserObjectRecordCollection;
BEGIN
	OPEN result FOR
		SELECT DISTINCT l.id, l.instance_of, object.id, object.instance_of FROM Link l
			INNER JOIN user_object ON user_object.id=l.owner
			INNER JOIN short_transaction ON short_transaction.id=l.created
			INNER JOIN long_transaction ON long_transaction.id=short_transaction.container
			INNER JOIN object ON object.id=user_object.id
		WHERE l.target=o
			AND l.instance_of=a
			AND objectfacade.rightdategreaterequalsleftdate(long_transaction.published,sysdate)=1
			AND l.id NOT IN
			(SELECT Link.id FROM Link INNER JOIN accessed ON accessed.v=Link.id
				INNER JOIN short_transaction ON short_transaction.id=accessed.t
				INNER JOIN long_transaction ON long_transaction.id=short_transaction.container
			WHERE accessed.v=l.id
				AND accessed.type='D'
				AND 
				(objectfacade.rightdategreaterequalsleftdate(long_transaction.published,sysdate)=1 OR long_transaction.id=trans))
		UNION
			SELECT DISTINCT Link.id, Link.instance_of, object.id, object.instance_of FROM Link
				INNER JOIN user_object ON user_object.id=Link.owner
				INNER JOIN short_transaction ON short_transaction.id=Link.created
				INNER JOIN object ON object.id=user_object.id
			WHERE Link.target=o
				AND Link.instance_of=a
				AND short_transaction.container=trans;   
	RETURN RESULT;
END inverseGetTransactionCursor;

/*
	inverseGetMapTransactionCursor(o NUMBER, a NUMBER, trans NUMBER) RETURN UserObjectRecordCollection
	Die Hilfsfunktion inverseGetTransactionCursor liefert einen offenen Cursor der, alle Links und ihre Owner-UserObjects, die das UserObject mit der Id <o> als Target und den map-key der Id <key> haben,
	von der Assoziation <a> sind und innerhalb der langen Transaktion mit der Id <trans> erstellt und nicht gelscht sind enthlt.
*/
FUNCTION inverseGetMapTransactionCursor(o NUMBER, a NUMBER, trans NUMBER) RETURN ObjectKeyLinkRecordCollection IS 
	result ObjectKeyLinkRecordCollection;
BEGIN
	OPEN result FOR
		SELECT DISTINCT object.id, object.instance_of, MapLink.map_key, MapAssociation.map_key_type, l.id, l.instance_of FROM Link l
			INNER JOIN MapLink ON MapLink.id=l.id
			INNER JOIN user_object ON user_object.id=l.owner
			INNER JOIN short_transaction ON short_transaction.id=l.created
			INNER JOIN long_transaction ON long_transaction.id=short_transaction.container
			INNER JOIN object ON object.id=user_object.id
			INNER JOIN MapAssociation ON MapAssociation.id=l.instance_of
		WHERE l.target=o
			AND l.instance_of=a
			AND objectfacade.rightdategreaterequalsleftdate(long_transaction.published,sysdate)=1
			AND l.id NOT IN
			(SELECT Link.id FROM Link INNER JOIN accessed ON accessed.v=Link.id
				INNER JOIN short_transaction ON short_transaction.id=accessed.t
				INNER JOIN long_transaction ON long_transaction.id=short_transaction.container
			WHERE accessed.v=l.id
				AND accessed.type='D'
				AND 
				(objectfacade.rightdategreaterequalsleftdate(long_transaction.published,sysdate)=1 OR long_transaction.id=trans))
		UNION
			SELECT DISTINCT object.id, object.instance_of, ltd.map_key, MapAssociation.map_key_type, Link.id, Link.instance_of FROM Link
				INNER JOIN MapLink ltd ON ltd.id=Link.id
				INNER JOIN user_object ON user_object.id=Link.owner
				INNER JOIN short_transaction ON short_transaction.id=Link.created
				INNER JOIN object ON object.id=user_object.id
				INNER JOIN MapAssociation ON MapAssociation.id=Link.instance_of
			WHERE Link.target=o
				AND Link.instance_of=a
				AND short_transaction.container=trans;   
	RETURN RESULT;
END inverseGetMapTransactionCursor;


FUNCTION getConflictingTransactions(tNumber NUMBER) RETURN TransactionIDCollection IS
	result TransactionIDCollection;
BEGIN
	OPEN result FOR 
		SELECT DISTINCT l.id FROM long_transaction l 
			INNER JOIN short_transaction s ON s.container=l.id 
			INNER JOIN accessed a ON a.t=s.id 
			WHERE published IS NULL AND l.id <> tNumber AND a.v IN
				(SELECT a2.v FROM long_transaction l2 
					INNER JOIN short_transaction s2 ON s2.container=l2.id 
					INNER JOIN accessed a2 ON a2.t=s2.id 
				WHERE  a2.type IN ('M','D') AND l2.id=tNumber);
	RETURN result;
END;
  
PROCEDURE rollbackTransaction(tNumber Number) IS  
	help NUMBER;
BEGIN
	/* check if transaction IS open */
	DELETE FROM Link
		WHERE created IN (SELECT id FROM short_transaction WHERE container=tNumber);
	DELETE FROM object
		WHERE id IN (SELECT o.id FROM user_object uo INNER JOIN object o ON o.id=uo.id INNER JOIN short_transaction s ON s.id=uo.created WHERE s.container=tNumber);
	DELETE FROM user_object
		WHERE created  IN (SELECT id FROM short_transaction WHERE container=tNumber);
	DELETE FROM Accessed
		WHERE t IN (SELECT id FROM short_transaction WHERE container=tNumber);
	/* check if user object IS transaction AND roll it back then */
	DELETE FROM short_transaction
		WHERE id IN (SELECT id FROM short_transaction WHERE container=tNumber);
	SELECT is_adhoc INTO help FROM long_transaction
		WHERE id=tNumber;
		IF (help=1) THEN
			DELETE FROM long_transaction
				WHERE id=tNumber;
		END IF;
END;

PROCEDURE commitTransaction(tNumber Number) IS
	toRollback TransactionIDCollection;
	t_row TransactionID;
BEGIN
	toRollback := getConflictingTransactions(tNumber);
	LOOP
		FETCH toRollback INTO t_row;
		EXIT WHEN toRollback%NOTFOUND;
		rollbackTransaction(t_row.id);
	END LOOP;
	CLOSE toRollback;
	UPDATE long_transaction SET published=SYSDATE WHERE id=tNumber;
END;

/*
	unset(l NUMBER, trans NUMBER)
	Die Prozedur unset lscht den Link mit der Id <l>, wenn die lange Transaktion mit der Id <trans> den Link mit der Id <l> erstellt hat.
	Wenn die lange Transaktion mit der Id <trans> den Link mit der Id <l> nicht erstellt hat, dann wird ein gelscht-Accessed Vermerk fr <l> eingefgt.
	Ist die Assoziation von dem Link mit der Id <l> Essential, dann wird ein modified-Accessed Vermerk auf dem Owner des Links mit der Id <l> eingefgt.
*/
PROCEDURE unset(l NUMBER, trans NUMBER)  IS 
	essential NUMBER;
	owner NUMBER;
BEGIN
	IF linkCreatedBy(l)=trans THEN
		DELETE FROM MapLink
			WHERE MapLink.id=l;
		DELETE FROM Link
			WHERE Link.id=l;		
	ELSE
		objectfacade.insertDeletedAccessed(l, trans);
		SELECT Association.essential INTO essential FROM Link
			INNER JOIN Association ON Association.id=Link.instance_of
				WHERE Link.id=l;
		IF essential=1 THEN
			SELECT user_object.id INTO owner FROM Link
				INNER JOIN user_object ON user_object.id=Link.owner
			WHERE Link.id=l;
			objectfacade.insertModifiedAccessed(owner, trans);
		END IF;
	END IF;
END unset;

FUNCTION rightDateGreaterEqualsLeftDate(leftDate TIMESTAMP, rightDate TIMESTAMP) RETURN NUMBER IS
BEGIN
	IF rightDate IS NULL THEN
		RETURN 1;
	END IF;
	IF leftDate IS NULL THEN
		RETURN 0;
	END IF;
	If leftDate <= rightDate THEN
		RETURN 1;
	END IF;
	RETURN 0;
END;
  
FUNCTION isTransactionOpen(tNumber Number) RETURN NUMBER IS
	result NUMBER;
BEGIN
	SELECT count(*) INTO result FROM long_transaction
		WHERE id=tNumber AND published IS NULL;
	IF (result=0) THEN
		RETURN 0;
	ELSE
		RETURN 1;
	END IF;
END;
    
/*  Die Funktion createUserObject erstellt innerhalb von <longTA> eine 
    Transaktion oder ein UserObject vom Typ <userType>. Falls der Typ in
    <userType> nicht existiert, wird eine <no_user_type_exception> geraised.
*/
FUNCTION createUserObject(userType NUMBER, longTA NUMBER) RETURN NUMBER  IS  
	result IDRecord;
	trans NUMBER;
	valid_type NUMBER;
	next_id NUMBER;
	short_ta NUMBER;
BEGIN
	SELECT count(*) INTO valid_type FROM type 
		WHERE id=userType AND abstract=0;
	IF (valid_type <> 1) THEN
		RAISE userexception.no_user_type_exception;
	END IF;

	SELECT InstanceItem_Seq.nextval INTO next_id FROM dual;
	SELECT trans INTO trans FROM type WHERE id=userType;

	INSERT INTO object VALUES (next_id,userType);
	INSERT INTO user_object VALUES (next_id,getActiveShortTransaction(longTA));

	IF (trans=1) THEN
		INSERT INTO long_transaction VALUES (next_id,0,NULL);
		INSERT INTO short_transaction VALUES (short_transaction_seq.nextval,next_id, NULL);
	END IF;

	result.id := next_id;
	result.classid := userType;
	RETURN result.id;
END createUserObject;
  
/*  Die Funktion deleteUserObject lscht ein UserObject, falls 
    <userObjectToBeDeleted> fr <LongTA> sichtbar ist und es keine Links gibt,
    die <userObjectToBeDeleted> als owner oder target referenzieren.
    Falls <userObjectToBeDeleted> von <LongTA> erstellt wurde, wird der
    Datensatz pysikalisch aus der Datenbank gelscht. Anderenfalls wird fr
    <userObjectToBeDeleted> ein Accessed-'D'-Eintrag erstellt.
*/
PROCEDURE deleteUserObject(userObjectToBeDeleted NUMBER, associatedLongTA NUMBER) IS
BEGIN
	IF (isUOInScope(userObjectToBeDeleted, associatedLongTA)) THEN
		IF (remainingLinks(userObjectToBeDeleted, associatedLongTA)>=1) THEN
			RAISE userexception.link_exception;
		ELSE
			IF (userObjectCreatedBy(userObjectToBeDeleted)=associatedLongTA) THEN
				DELETE FROM object
					WHERE id=userObjectToBeDeleted;
				DELETE FROM user_object
					WHERE id=userObjectToBeDeleted;
			ELSE
				InsertDeletedAccessed(userObjectToBeDeleted, associatedLongTA);
			END IF;
		END IF;
	END IF;
END deleteUserObject;
  
/* 
	Die Funktion makeSavePoint erstellt eine neue aktive kurze Transaktion fr die lange Transaktion mit der Id <tNumber>.
*/
PROCEDURE makeSavePoint(tNumber NUMBER) IS
	next_id NUMBER;
BEGIN
	/* Check if transaction open */
	SELECT short_transaction_seq.nextval INTO next_id FROM dual;
	INSERT INTO short_transaction VALUES (next_id, tNumber, NULL);
	UPDATE short_transaction SET next=next_id 
		WHERE container=tNumber AND id<>next_id AND next IS NULL;
	END;

PROCEDURE rollbackToSavePoint(tNumber Number) IS
	prev_short_ta NUMBER;
	last_short_ta NUMBER;
	helper NUMBER;
BEGIN
	SELECT id INTO last_short_ta FROM short_transaction
		WHERE next IS NULL AND container=tNumber;
	/* check if transaction IS open */
	DELETE FROM Link
		WHERE created=last_short_ta;
	DELETE FROM object
		WHERE id IN (SELECT id FROM user_object WHERE created=last_short_ta);
	DELETE FROM user_object
		WHERE created=last_short_ta;
	/* check if user object IS transaction AND roll them back */
	DELETE FROM accessed
		WHERE t=last_short_ta;
	SELECT count(*) INTO helper FROM short_transaction
		WHERE container=tNumber AND id<>lasT_short_ta;
	IF (helper=0) THEN
		DELETE FROM short_transaction
			WHERE id=last_short_ta;
		SELECT short_transaction_seq.nextval INTO last_short_ta FROM dual;
		INSERT INTO short_transaction(id,next,container) values (last_short_ta, NULL, tNumber);
	ELSE
		SELECT id INTO prev_short_ta FROM short_transaction WHERE next=last_short_ta;
		UPDATE short_transaction SET next=null
			WHERE id=prev_short_ta;
		DELETE FROM short_transaction
			WHERE id=last_short_ta;
	END IF;
END;

/*
	Die Funktion getString liefert den String mit der Id <s>.
	Falls kein String mit der Id <s> existiert, wird eine base_type_not_found_exception graised.
*/
FUNCTION getString(s NUMBER) RETURN VARCHAR2 IS
	result VARCHAR2(4000);
	help NUMBER;
BEGIN
	SELECT count(*) INTO help FROM string_value
		WHERE id=s;
	IF (help=0) then
		raise userexception.base_type_not_found_exception;
	end if;
	SELECT string_value.value INTO result FROM string_value
		WHERE string_value.id=s;
	RETURN result;
END getString;
  
/*
	Die Funktion getInteger liefert den Integer mit der Id <i>.
	Falls kein Iteger mit der Id <i> existiert, wird eine base_type_not_found_exception graised.
*/
FUNCTION getInteger(i NUMBER) RETURN NUMBER IS
	result NUMBER;
	help NUMBER;
BEGIN
	SELECT count(*) INTO help FROM integer_value
		WHERE id=i;
	IF (help=0) then
		raise userexception.base_type_not_found_exception;
	end if;
	SELECT integer_value.value INTO result FROM integer_value
		WHERE integer_value.id=i;
RETURN result;
END getInteger;
  
/*
	Die Funktion provideAdhocTransaction erstellt eine neue Adhoc-Transaktion zusammen mit einer dazugehrigen ersten bzw. aktiven kurzen Transaktion.
*/
FUNCTION provideAdhocTransaction RETURN NUMBER IS
	result NUMBER;
BEGIN
	SELECT instanceitem_seq.nextval INTO result FROM dual;
		INSERT INTO long_transaction(id,is_adhoc, published) values (result,1,NULL);
		INSERT INTO short_transaction(id,container,next) values (short_transaction_seq.nextval,result,NULL);
	RETURN result;
END;  

PROCEDURE clear IS
BEGIN
DELETE FROM accessed;
DELETE FROM integer_value;
DELETE FROM MapLink;
DELETE FROM Link;
DELETE FROM object;
DELETE FROM user_object;
DELETE FROM short_transaction;
DELETE FROM long_transaction;
DELETE FROM string_value;

END;

end objectfacade;
