/*
 * Copyright © 2020 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of gaming-gui.
 *
 * Gaming-gui is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * Gaming-gui is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * gaming-gui. If not, see <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.gui;

import java.util.Optional;
import java.util.Set;

import de.fhdw.gaming.core.domain.Game;
import de.fhdw.gaming.core.domain.Move;
import de.fhdw.gaming.core.domain.Observer;
import de.fhdw.gaming.core.domain.Player;
import de.fhdw.gaming.core.domain.State;
import javafx.application.Platform;
import javafx.scene.control.ListView;

/**
 * Implements {@link Observer} by logging all events.
 */
final class LogObserver implements Observer {

    /**
     * The Singleton instance of this class.
     */
    public static final LogObserver INSTANCE = new LogObserver();

    /**
     * The label to use for displaying events.
     */
    private Optional<ListView<String>> label;

    /**
     * Private constructor for Singleton pattern.
     */
    private LogObserver() {
        this.label = Optional.empty();
    }

    /**
     * Sets the log to use for displaying events.
     *
     * @param log The log to use for displaying events.
     */
    public void setLog(final ListView<String> log) {
        this.label = Optional.ofNullable(log);
    }

    @Override
    public void started(final Game<?, ?, ?, ?> game, final State<?, ?> state) {
        Platform.runLater(() -> this.label.ifPresent((final ListView<String> l) -> l.getItems().add("Game started.")));
    }

    @Override
    public void nextPlayersComputed(final Game<?, ?, ?, ?> game, final State<?, ?> state,
            final Set<? extends Player<?>> players) {
        Platform.runLater(
                () -> this.label.ifPresent(
                        (final ListView<String> l) -> l.getItems().add(String.format("Next player(s): %s.", players))));
    }

    @Override
    public void illegalPlayerRejected(final Game<?, ?, ?, ?> game, final State<?, ?> state, final Player<?> player) {
        Platform.runLater(
                () -> this.label.ifPresent(
                        (final ListView<String> l) -> l.getItems()
                                .add(String.format("Illegal player rejected: %s.", player))));
    }

    @Override
    public void legalMoveApplied(final Game<?, ?, ?, ?> game, final State<?, ?> state, final Player<?> player,
            final Move<?, ?> move) {
        Platform.runLater(
                () -> this.label.ifPresent(
                        (final ListView<String> l) -> l.getItems()
                                .add(String.format("Move applied for player %s: %s.", player, move))));
    }

    @Override
    public void illegalMoveRejected(final Game<?, ?, ?, ?> game, final State<?, ?> state, final Player<?> player,
            final Optional<Move<?, ?>> move, final String reason) {
        Platform.runLater(
                () -> this.label.ifPresent(
                        (final ListView<String> l) -> l.getItems().add(
                                String.format("Move rejected for player %s: %s. Reason: %s.", player, move, reason))));
    }

    @Override
    public void overdueMoveRejected(final Game<?, ?, ?, ?> game, final State<?, ?> state, final Player<?> player,
            final Optional<Move<?, ?>> chosenMove) {
        Platform.runLater(
                () -> this.label.ifPresent(
                        (final ListView<String> l) -> l.getItems().add(
                                String.format(
                                        "Overdue move rejected for player %s. Chosen move: %s.",
                                        player,
                                        chosenMove))));
    }

    @Override
    public void playerResigned(final Game<?, ?, ?, ?> game, final State<?, ?> state, final Player<?> player) {
        Platform.runLater(
                () -> this.label.ifPresent(
                        (final ListView<String> l) -> l.getItems()
                                .add(String.format("Player<?> %s has resigned.", player))));
    }

    @Override
    public void playerOvertaken(final Game<?, ?, ?, ?> game, final State<?, ?> state, final Player<?> overtakenPlayer,
            final Player<?> overtakingPlayer) {
        Platform.runLater(
                () -> this.label.ifPresent(
                        (final ListView<String> l) -> l.getItems().add(
                                String.format(
                                        "Player %s has been overtaken by player %s.",
                                        overtakenPlayer,
                                        overtakingPlayer))));
    }

    @Override
    public void finished(final Game<?, ?, ?, ?> game, final State<?, ?> state) {
        final String gameResult = getGameResult(game);
        Platform.runLater(
                () -> this.label.ifPresent(
                        (final ListView<String> l) -> l.getItems().add(
                                String.format(
                                        "Game finished.%s",
                                        gameResult.isEmpty() ? "" : String.format(" Result:%s", gameResult)))));
    }

    /**
     * Computes the game result for the log message.
     *
     * @param game The game.
     * @return A message summarising the result of the game.
     */
    private static String getGameResult(final Game<?, ?, ?, ?> game) {
        final StringBuilder sb = new StringBuilder();
        for (final Player<?> player : game.getPlayers().values()) {
            sb.append(" Player ").append(player).append('.');
        }
        return sb.toString();
    }
}
