package de.fhdw.gaming.gui.util;

import java.awt.Dimension;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.io.InputStream;
import java.util.Iterator;
import java.util.Optional;

import javax.imageio.ImageIO;
import javax.imageio.ImageReadParam;
import javax.imageio.ImageReader;
import javax.imageio.stream.ImageInputStream;

import javafx.beans.value.ObservableBooleanValue;
import javafx.beans.value.ObservableValue;
import javafx.embed.swing.SwingFXUtils;
import javafx.scene.control.Alert;
import javafx.scene.control.Alert.AlertType;
import javafx.scene.control.Button;
import javafx.scene.control.ButtonType;
import javafx.scene.control.ContentDisplay;
import javafx.scene.control.Tooltip;
import javafx.scene.image.Image;
import javafx.scene.image.ImageView;
import javafx.scene.layout.Region;
import javafx.stage.Window;

/**
 * Contains JavaFX utility operations.
 */
public final class FXUtil {

    /**
     * The part of the available space the button icon shall occupy.
     */
    private static final double ICON_RATIO = 0.75;
    /**
     * The minimum button width.
     */
    private static final double MIN_BUTTON_WIDTH = 50.0;
    /**
     * The minimum button height.
     */
    private static final double MIN_BUTTON_HEIGHT = 50.0;

    /**
     * Describes a button.
     */
    public static final class ButtonDescriptor {

        /**
         * The button text.
         */
        private String buttonText;
        /**
         * The button description.
         */
        private String buttonDescription;
        /**
         * The icon name.
         */
        private String iconName;

        /**
         * Returns the button text.
         */
        public String getButtonText() {
            return this.buttonText;
        }

        /**
         * Returns the button description.
         */
        public String getButtonDescription() {
            return this.buttonDescription;
        }

        /**
         * Returns the icon name.
         */
        public String getIconName() {
            return this.iconName;
        }

        /**
         * Changes the button text.
         *
         * @param newButtonText The new button text.
         * @return {@code this}
         */
        public ButtonDescriptor text(final String newButtonText) {
            this.buttonText = newButtonText;
            return this;
        }

        /**
         * Changes the button description.
         *
         * @param newButtonDescription The new button description.
         * @return {@code this}
         */
        public ButtonDescriptor description(final String newButtonDescription) {
            this.buttonDescription = newButtonDescription;
            return this;
        }

        /**
         * Changes the icon name.
         *
         * @param newIconName The new icon name.
         * @return {@code this}
         */
        public ButtonDescriptor icon(final String newIconName) {
            this.iconName = newIconName;
            return this;
        }
    }

    /**
     * Private constructor for utility class.
     */
    private FXUtil() {
    }

    /**
     * Displays an alert dialog.
     *
     * @param owner    The owner window.
     * @param type     The alert type.
     * @param contents The message to display.
     * @return The button chosen (if any).
     */
    public static Optional<ButtonType> showAlert(final Window owner, final AlertType type, final String contents) {
        final Alert alert = new Alert(type, contents);
        alert.initOwner(owner); // dialog is centred on owner
        alert.setResizable(true); // needed especially for GTK+
        alert.getDialogPane().setMinHeight(Region.USE_PREF_SIZE); // expand dialog to make contents visible
        return alert.showAndWait();
    }

    /**
     * Converts an SVG input stream into an {@link Image}.
     *
     * @param in     The input stream. Must be closed by the caller.
     * @param width  The desired width of the image.
     * @param height The desired height of the image.
     * @return The {@link Image}. May be empty if an error occurred while loading or parsing the SVG data.
     */
    public static Optional<Image> fromSVGImage(final InputStream in, final double width, final double height) {
        try (ImageInputStream imageInputStream = ImageIO.createImageInputStream(in)) {
            final Iterator<ImageReader> itReader = ImageIO.getImageReaders(imageInputStream);
            if (!itReader.hasNext()) {
                return Optional.empty();
            }

            final ImageReader reader = itReader.next();
            final ImageReadParam param = reader.getDefaultReadParam();
            if (param.canSetSourceRenderSize()) {
                param.setSourceRenderSize(new Dimension((int) width, (int) height));
            }
            reader.setInput(imageInputStream, true, true);

            try {
                final BufferedImage bufferedImage = reader.read(0, param);
                return Optional.of(SwingFXUtils.toFXImage(bufferedImage, null));
            } finally {
                reader.dispose();
            }
        } catch (final IOException e) {
            return Optional.empty();
        }
    }

    /**
     * Creates a button with an SVG image.
     *
     * @param buttonDescriptor The button description.
     * @return The button.
     */
    public static Button createButtonWithSVGImage(final ButtonDescriptor buttonDescriptor) {
        final Button button = new Button();
        button.setContentDisplay(ContentDisplay.TOP);
        button.setMaxSize(Double.MAX_VALUE, Double.MAX_VALUE);
        button.setMinSize(FXUtil.MIN_BUTTON_WIDTH, FXUtil.MIN_BUTTON_HEIGHT);
        setButtonProperties(button, buttonDescriptor, 1.0, 1.0);

        button.widthProperty().addListener(
                (final ObservableValue<? extends Number> value, final Number oldValue, final Number newValue) ->
                    loadAndSetButtonIcon(
                            button,
                            buttonDescriptor.getIconName(),
                            newValue.doubleValue() * FXUtil.ICON_RATIO,
                            button.getHeight() * FXUtil.ICON_RATIO));
        button.heightProperty().addListener(
                (final ObservableValue<? extends Number> value, final Number oldValue, final Number newValue) ->
                    loadAndSetButtonIcon(
                            button,
                            buttonDescriptor.getIconName(),
                            button.getWidth() * FXUtil.ICON_RATIO,
                            newValue.doubleValue() * FXUtil.ICON_RATIO));
        return button;
    }

    /**
     * Creates a two-mode button with an SVG image.
     *
     * @param mode                      The {@link ObservableBooleanValue} denoting the button's mode.
     * @param buttonDescriptorModeFalse The button descriptor if {@code mode} is {@code false}.
     * @param buttonDescriptorModeTrue  The button descriptor if {@code mode} is {@code true}.
     * @return The button.
     */
    public static Button createTwoModeButtonWithSVGImage(final ObservableBooleanValue mode,
            final ButtonDescriptor buttonDescriptorModeFalse, final ButtonDescriptor buttonDescriptorModeTrue) {

        final Button button = new Button();
        button.setContentDisplay(ContentDisplay.TOP);
        button.setMaxSize(Double.MAX_VALUE, Double.MAX_VALUE);
        button.setMinSize(FXUtil.MIN_BUTTON_WIDTH, FXUtil.MIN_BUTTON_HEIGHT);
        setButtonProperties(button, mode.get() ? buttonDescriptorModeTrue : buttonDescriptorModeFalse, 1.0, 1.0);

        mode.addListener(
                (final ObservableValue<? extends Boolean> value, final Boolean oldValue, final Boolean newValue) ->
                    setButtonProperties(
                            button,
                            newValue ? buttonDescriptorModeTrue : buttonDescriptorModeFalse,
                            button.getWidth() * FXUtil.ICON_RATIO,
                            button.getHeight() * FXUtil.ICON_RATIO));
        button.widthProperty().addListener(
                (final ObservableValue<? extends Number> value, final Number oldValue, final Number newValue) ->
                    setButtonProperties(
                            button,
                            mode.get() ? buttonDescriptorModeTrue : buttonDescriptorModeFalse,
                            newValue.doubleValue() * FXUtil.ICON_RATIO,
                            button.getHeight() * FXUtil.ICON_RATIO));
        button.heightProperty().addListener(
                (final ObservableValue<? extends Number> value, final Number oldValue, final Number newValue) ->
                    setButtonProperties(
                            button,
                            mode.get() ? buttonDescriptorModeTrue : buttonDescriptorModeFalse,
                            button.getWidth() * FXUtil.ICON_RATIO,
                            newValue.doubleValue() * FXUtil.ICON_RATIO));
        return button;
    }

    /**
     * Sets the properties of the button.
     *
     * @param button           The button.
     * @param buttonDescriptor The button descriptor.
     * @param width            The width to set.
     * @param height           The height to set.
     */
    private static void setButtonProperties(final Button button, final ButtonDescriptor buttonDescriptor,
            final double width, final double height) {
        button.setText(buttonDescriptor.getButtonText());
        button.setTooltip(new Tooltip(buttonDescriptor.getButtonDescription()));
        loadAndSetButtonIcon(button, buttonDescriptor.getIconName(), width, height);
    }

    /**
     * Loads and sets the icon of a button.
     *
     * @param button   The button.
     * @param iconName The icon name.
     * @param width    The desired width.
     * @param height   The desired height.
     */
    private static void loadAndSetButtonIcon(final Button button, final String iconName, final double width,
            final double height) {
        final String path = String.format("/icons/%s.svg", iconName);
        try (InputStream in = FXUtil.class.getResourceAsStream(path)) {
            final Optional<Image> buttonImage = FXUtil.fromSVGImage(in, Math.max(1.0, width), Math.max(1.0, height));
            buttonImage.ifPresent((final Image image) -> button.setGraphic(new ImageView(image)));
        } catch (final IOException e) {
            button.setGraphic(null);
        }
    }
}
