/*
 * Copyright © 2020-2023 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of gaming-gui.
 *
 * Gaming-gui is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * Gaming-gui is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * gaming-gui. If not, see <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.gui;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;

import de.fhdw.gaming.core.ui.InputProvider;
import de.fhdw.gaming.core.ui.InputProviderException;
import de.fhdw.gaming.core.ui.type.BooleanFieldType;
import de.fhdw.gaming.core.ui.type.FieldType;
import de.fhdw.gaming.core.ui.type.FieldTypeException;
import de.fhdw.gaming.core.ui.type.IntegerFieldType;
import de.fhdw.gaming.core.ui.type.ObjectFieldType;
import de.fhdw.gaming.core.ui.type.StringFieldType;
import de.fhdw.gaming.core.ui.type.validator.Validator;
import javafx.animation.KeyFrame;
import javafx.animation.Timeline;
import javafx.application.Platform;
import javafx.beans.property.SimpleObjectProperty;
import javafx.beans.value.WritableObjectValue;
import javafx.event.ActionEvent;
import javafx.geometry.HPos;
import javafx.geometry.Insets;
import javafx.scene.control.ButtonType;
import javafx.scene.control.Dialog;
import javafx.scene.control.Label;
import javafx.scene.layout.Background;
import javafx.scene.layout.BackgroundFill;
import javafx.scene.layout.ColumnConstraints;
import javafx.scene.layout.CornerRadii;
import javafx.scene.layout.GridPane;
import javafx.scene.layout.Priority;
import javafx.scene.layout.Region;
import javafx.scene.paint.Color;
import javafx.stage.Window;
import javafx.util.Duration;

/**
 * An interactive {@link InputProvider} using JavaFX dialogs.
 */
final class InteractiveDialogInputProvider implements InputProvider {

    /**
     * The window owning the dialogs this input provider creates.
     */
    private final Window owner;
    /**
     * The needed data.
     */
    private final Map<String, NeededDataEntry> neededData;
    /**
     * The predefined data.
     */
    private final Map<String, Object> providedData;

    /**
     * Represents a needed data entry.
     */
    private static final class NeededDataEntry {

        /**
         * The prompt.
         */
        private final String prompt;
        /**
         * The field type.
         */
        private final FieldType<?> fieldType;

        /**
         * Creates a {@link NeededDataEntry}.
         *
         * @param prompt    The prompt.
         * @param fieldType The field type.
         */
        NeededDataEntry(final String prompt, final FieldType<?> fieldType) {
            this.prompt = prompt + ":";
            this.fieldType = fieldType;
        }

        /**
         * Returns the prompt.
         */
        String getPrompt() {
            return this.prompt;
        }

        /**
         * Returns the field type.
         */
        FieldType<?> getFieldType() {
            return this.fieldType;
        }
    }

    /**
     * Creates a {@link InteractiveDialogInputProvider}.
     *
     * @param owner The window owning the dialogs this input provider creates.
     */
    InteractiveDialogInputProvider(final Window owner) {
        this.owner = owner;
        this.neededData = new LinkedHashMap<>();
        this.providedData = new LinkedHashMap<>();
    }

    @Override
    @SafeVarargs
    public final InteractiveDialogInputProvider needString(final String id, final String prompt,
            final Optional<String> defaultValue, final Validator<String>... validators) {

        this.neededData.put(id, new NeededDataEntry(prompt, new StringFieldType(defaultValue).validateBy(validators)));
        return this;
    }

    @Override
    public InteractiveDialogInputProvider fixedString(final String id, final String fixedValue) {
        this.providedData.put(id, fixedValue);
        return this;
    }

    @Override
    @SafeVarargs
    public final InteractiveDialogInputProvider needInteger(final String id, final String prompt,
            final Optional<Integer> defaultValue, final Validator<Integer>... validators) {

        this.neededData.put(id, new NeededDataEntry(prompt, new IntegerFieldType(defaultValue).validateBy(validators)));
        return this;
    }

    @Override
    public InteractiveDialogInputProvider fixedInteger(final String id, final Integer fixedValue) {
        this.providedData.put(id, fixedValue);
        return this;
    }

    @Override
    @SafeVarargs
    public final InteractiveDialogInputProvider needBoolean(final String id, final String prompt,
            final Optional<Boolean> defaultValue, final Validator<Boolean>... validators) {

        this.neededData.put(id, new NeededDataEntry(prompt, new BooleanFieldType(defaultValue).validateBy(validators)));
        return this;
    }

    @Override
    public InteractiveDialogInputProvider fixedBoolean(final String id, final Boolean fixedValue) {
        this.providedData.put(id, fixedValue);
        return this;
    }

    @Override
    public InteractiveDialogInputProvider needObject(final String id, final String prompt,
            final Optional<Object> defaultValue, final Set<?> objectSet) {

        this.neededData.put(id, new NeededDataEntry(prompt, new ObjectFieldType(defaultValue, objectSet)));
        return this;
    }

    @Override
    public InteractiveDialogInputProvider fixedObject(final String id, final Object fixedValue) {
        this.providedData.put(id, fixedValue);
        return this;
    }

    @Override
    public Map<String, Object> requestData(final String title) throws InputProviderException {
        final Dialog<ButtonType> dialog = new Dialog<>();
        dialog.initOwner(this.owner);
        dialog.setTitle(title);
        dialog.setResizable(true);
        dialog.getDialogPane().getButtonTypes().add(ButtonType.OK);
        dialog.getDialogPane().getButtonTypes().add(ButtonType.CANCEL);

        final GridPane gridPane = new GridPane();
        gridPane.setHgap(10);
        gridPane.setVgap(10);

        final ColumnConstraints column1 = new ColumnConstraints();
        column1.setHalignment(HPos.RIGHT);
        column1.setHgrow(Priority.NEVER);
        gridPane.getColumnConstraints().add(column1);
        final ColumnConstraints column2 = new ColumnConstraints();
        column2.setHalignment(HPos.LEFT);
        column2.setHgrow(Priority.ALWAYS);
        column2.setMaxWidth(Double.MAX_VALUE);
        gridPane.getColumnConstraints().add(column2);

        dialog.getDialogPane().setContent(gridPane);

        final Map<String, WritableObjectValue<Object>> values = new LinkedHashMap<>();
        final Map<String, Region> controls = new LinkedHashMap<>();

        for (final Map.Entry<String, NeededDataEntry> entry : this.neededData.entrySet()) {
            final Region control = this
                    .addFieldControl(entry.getKey(), entry.getValue(), values, gridPane, controls.isEmpty());
            controls.put(entry.getKey(), control);
        }

        dialog.getDialogPane().lookupButton(ButtonType.OK)
                .addEventFilter(ActionEvent.ACTION, (final ActionEvent event) -> {
                    for (final Map.Entry<String, WritableObjectValue<Object>> entry : values.entrySet()) {
                        final Region control = controls.get(entry.getKey());
                        if (entry.getValue().get() == null) {
                            this.validationFailed(control);
                            event.consume();
                        }
                    }
                });

        final Optional<ButtonType> dialogResult = dialog.showAndWait();
        if (dialogResult.orElse(ButtonType.CANCEL).equals(ButtonType.OK)) {
            final Map<String, Object> result = new LinkedHashMap<>();
            for (final Map.Entry<String, WritableObjectValue<Object>> entry : values.entrySet()) {
                result.put(entry.getKey(), entry.getValue().get());
            }
            return result;
        } else {
            throw new InputProviderException("Dialog aborted.");
        }
    }

    /**
     * Creates and adds the controls for a field.
     *
     * @param id              The ID of the field.
     * @param neededDataEntry The {@link NeededDataEntry} associated with the field.
     * @param values          The map receiving the value of the field.
     * @param gridPane        The grid pane to add the control to.
     * @param requestFocus    If {@code true}, the field receives the focus initially.
     *
     * @return The control created that receives the input from the user.
     */
    private Region addFieldControl(final String id, final NeededDataEntry neededDataEntry,
            final Map<String, WritableObjectValue<Object>> values, final GridPane gridPane, final boolean requestFocus)
            throws InputProviderException {

        values.put(id, new SimpleObjectProperty<>(neededDataEntry.getFieldType().getDefaultValue().orElse(null)));

        final Label prompt = new Label(neededDataEntry.getPrompt());
        final Label errorLabel = new Label();
        errorLabel.setTextFill(Color.RED);
        final Consumer<String> validationHandler = (final String message) -> {
            errorLabel.setText(message);
            if (message.isEmpty()) {
                prompt.setTextFill(Color.BLACK);
            } else {
                prompt.setTextFill(Color.RED);
            }
        };

        final Optional<Object> provided = Optional.ofNullable(this.providedData.get(id));
        final ControlCreator controlCreator = new ControlCreator(id, validationHandler, provided, values);
        try {
            neededDataEntry.getFieldType().accept(controlCreator);

            gridPane.addRow(gridPane.getRowCount(), prompt, controlCreator.getResult());

            gridPane.addRow(gridPane.getRowCount(), errorLabel);
            GridPane.setColumnSpan(errorLabel, 2);
            GridPane.setHalignment(errorLabel, HPos.LEFT);
            GridPane.setHgrow(errorLabel, Priority.ALWAYS);
        } catch (final FieldTypeException e) {
            throw new InputProviderException(
                    String.format("Error while creating control for field %s: %s", id, e.getMessage()),
                    e);
        }

        if (requestFocus) {
            Platform.runLater(() -> controlCreator.getFocusTarget().requestFocus());
        }

        return controlCreator.getResult();
    }

    /**
     * Gives the user feedback about a field where validation failed.
     *
     * @param control The control associated with the field.
     */
    private void validationFailed(final Region control) {
        final Background oldBackground = control.getBackground();
        final Background redBackground = new Background(new BackgroundFill(Color.RED, CornerRadii.EMPTY, Insets.EMPTY));
        final Timeline timeline = new Timeline(
                new KeyFrame(Duration.seconds(0.4), evt -> control.setBackground(redBackground)),
                new KeyFrame(Duration.seconds(0.8), evt -> control.setBackground(oldBackground)));
        timeline.setCycleCount(2);
        timeline.play();
    }

    @Override
    public InputProvider getNext(final Map<String, Object> lastDataSet) {
        return new InteractiveDialogInputProvider(this.owner);
    }
}
