package de.fhdw.gaming.contest;

import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.util.Optional;
import java.util.Set;

import de.fhdw.gaming.core.domain.Game;
import de.fhdw.gaming.core.domain.Move;
import de.fhdw.gaming.core.domain.Observer;
import de.fhdw.gaming.core.domain.Player;
import de.fhdw.gaming.core.domain.State;
import de.fhdw.gaming.core.domain.util.EventListCreator;

/**
 * Records all events.
 */
public final class RecordingObserver implements Observer, AutoCloseable {

    /**
     * The {@link File} object where to store the game events.
     */
    private final File gameEventFile;
    /**
     * The underlying event list creator.
     */
    private final EventListCreator eventListCreator;
    /**
     * The final state of the game.
     */
    private String finalState;

    /**
     * Constructor.
     *
     * @param gameEventFile The file where to store the game events.
     */
    public RecordingObserver(final File gameEventFile) {
        this.gameEventFile = gameEventFile;
        this.eventListCreator = new EventListCreator();
    }

    @Override
    public void started(final Game<?, ?, ?, ?> game, final State<?, ?> state) {
        this.eventListCreator.started(state);
    }

    @Override
    public void nextPlayersComputed(final Game<?, ?, ?, ?> game, final State<?, ?> state,
            final Set<? extends Player<?>> players) {
        this.eventListCreator.nextPlayersComputed(state, players);
    }

    @Override
    public void illegalPlayerRejected(final Game<?, ?, ?, ?> game, final State<?, ?> state, final Player<?> player) {
        this.eventListCreator.illegalPlayerRejected(state, player);
    }

    @Override
    public void legalMoveApplied(final Game<?, ?, ?, ?> game, final State<?, ?> state, final Player<?> player,
            final Move<?, ?> move) {
        this.eventListCreator.legalMoveApplied(state, player, move);
    }

    @Override
    public void illegalMoveRejected(final Game<?, ?, ?, ?> game, final State<?, ?> state, final Player<?> player,
            final Optional<Move<?, ?>> move, final String reason) {
        this.eventListCreator.illegalMoveRejected(state, player, move, reason);
    }

    @Override
    public void overdueMoveRejected(final Game<?, ?, ?, ?> game, final State<?, ?> state, final Player<?> player,
            final Optional<Move<?, ?>> chosenMove) {
        this.eventListCreator.overdueMoveRejected(state, player, chosenMove);
    }

    @Override
    public void playerResigned(final Game<?, ?, ?, ?> game, final State<?, ?> state, final Player<?> player) {
        this.eventListCreator.playerResigned(state, player);
    }

    @Override
    public void playerOvertaken(final Game<?, ?, ?, ?> game, final State<?, ?> state, final Player<?> overtakenPlayer,
            final Player<?> overtakingPlayer) {
        this.eventListCreator.playerOvertaken(state, overtakenPlayer, overtakingPlayer);
    }

    @Override
    public void finished(final Game<?, ?, ?, ?> game, final State<?, ?> state) {
        this.eventListCreator.finished(state);
        this.finalState = "FINAL GAME STATE\n================\n" + state.toString() + '\n';
    }

    @Override
    public void close() throws IOException {
        try (PrintStream out = new PrintStream(this.gameEventFile)) {
            out.println(this.eventListCreator.getResult());
            if (this.finalState != null) {
                out.println(this.finalState);
            }
        }
    }
}
