package de.fhdw.gaming.ipspiel21.dilemmaOriginal.strategy;

import java.util.List;
import java.util.Optional;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.domain.DilemmaPlayer;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.domain.DilemmaState;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.domain.DilemmaStrategy;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.moves.DilemmaMove;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.moves.factory.DilemmaMoveFactory;
import de.fhdw.gaming.ipspiel21.evolution.GameHistory;

/**
 * Implements {@link DilemmaStrategy} with the punisher strategy. Stays silent until the other player confesses for the
 * first time. Then confesses until both players have the same outcome.
 */
final class DilemmaPunisherStrategy implements DilemmaStrategy {

    /**
     * Factory for creating Dilemma moves.
     */
    private final DilemmaMoveFactory moveFactory;

    /**
     * Creates a {@link DilemmaTitForTatStrategy}.
     *
     * @param moveFactory The factory for creating Dilemma moves.
     */
    public DilemmaPunisherStrategy(final DilemmaMoveFactory moveFactory) {
        this.moveFactory = moveFactory;
    }

    @Override
    public Optional<DilemmaMove> computeNextMove(final int gameId, final DilemmaPlayer player, final DilemmaState state)
            throws GameException, InterruptedException {

        // Player should stay silent in the first round
        if (player.getGameHistoryCollection().numberOfPlayedGames() == 0) {
            return Optional.of(moveFactory.createYesMove());
        }

        if (player.getGameHistoryCollection().getSpecificGameHistory(0).getOpponentMove(0)
                .equals(moveFactory.createNoMove())
                || this.getOutcomeDifference(player.getGameHistoryCollection().getGameHistories()) < 0) {
            return Optional.of(moveFactory.createNoMove());
        } else {
            return Optional.of(moveFactory.createYesMove());
        }
    }

    /**
     * Return the difference of all outcomes between the opponent player and the player. Is positive, if the player has
     * a higher overall outcome than the opponent.
     * 
     * @param games List of all games that have been played until the current game
     */
    private double getOutcomeDifference(final List<GameHistory> games) {
        Double playerOutcome = 0.0;
        Double enemyPlayerOutcome = 0.0;
        for (final GameHistory game : games) {
            playerOutcome += game.getOutcome().get();
            enemyPlayerOutcome += game.getOpponentOutcome().get();
        }
        return playerOutcome - enemyPlayerOutcome;
    }

    @Override
    public String toString() {
        return DilemmaPunisherStrategy.class.getSimpleName();
    }
}
