package de.fhdw.gaming.ipspiel21.dilemmaOriginal.strategy;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Set;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.domain.DilemmaPlayer;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.domain.DilemmaState;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.domain.DilemmaStrategy;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.moves.DilemmaMove;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.moves.impl.DilemmaBeSilentMove;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.moves.impl.DilemmaConfessMove;
import de.fhdw.gaming.ipspiel21.evolution.GameHistoryCollection;

/**
 * Implements {@link DilemmaStrategy} by saying that what was set before the game started.
 */
public final class DilemmaCustomStrategy implements DilemmaStrategy {

    /**
     * Constant that represents the identifier when the opponent moves are {@link DilemmaConfessMove) for 100 percent.
     */
    private static final String IDENTIFIER_100_CONFESS = new DilemmaConfessMove().toString() + " Numberbased100";
    /**
     * Constant that represents the identifier when the opponent moves are {@link DilemmaConfessMove) for >50 percent.
     */
    private static final String IDENTIFIER_50_CONFESS = new DilemmaConfessMove().toString() + " Numberbased50";
    /**
     * Constant that represents the identifier when the opponent moves are {@link DilemmaBeSilentMove) for 100 percent.
     */
    private static final String IDENTIFIER_100_BESILENT = new DilemmaBeSilentMove().toString() + " Numberbased100";
    /**
     * Constant that represents the identifier when the opponent moves are {@link DilemmaBeSilentMove) for >50 percent.
     */
    private static final String IDENTIFIER_50_BESILENT = new DilemmaBeSilentMove().toString() + " Numberbased50";
    /**
     * The provided Data.
     */
    private final Map<String, DilemmaMove> providedMoveData = new LinkedHashMap<>();
    /**
     * Amount of considered games.
     */
    private Integer amountOfGames;
    /**
     * Default move of player.
     */
    private DilemmaMove initialMove;
    
    /**
     * Sets the provided moves.
     * 
     * @param providedMoveData
     */
    public void setProvidedMoveData(final Map<String, Object> providedMoveData) {
        final Set<Entry<String, Object>> entrySet = providedMoveData.entrySet();

        this.providedMoveData.clear();
        for (final Entry<String, Object> entry : entrySet) {
            if (IDENTIFIER_100_BESILENT.equals(entry.getKey()) || IDENTIFIER_100_CONFESS.equals(entry.getKey())
                    || IDENTIFIER_50_BESILENT.equals(entry.getKey()) || IDENTIFIER_50_CONFESS.equals(entry.getKey())) {
                this.providedMoveData.put(entry.getKey(), (DilemmaMove) entry.getValue());
            }
        }
    }

    /**
     * Sets the amount of considered games.
     * 
     * @param amountOfGames The amount of considered games.
     */
    public void setAmountOfGames(final Integer amountOfGames) {
        this.amountOfGames = amountOfGames;
    }

    /**
     * Sets the initial move.
     * 
     * @param initialMove The initial move choice of the player
     */
    public void setInitialMove(final DilemmaMove initialMove) {
        this.initialMove = initialMove;
    }

    @Override
    public Optional<DilemmaMove> computeNextMove(final int gameId, final DilemmaPlayer player, 
            final  DilemmaState state)
            throws GameException, InterruptedException {
        if (providedMoveData.isEmpty()) {
            return Optional.of(initialMove);
        }
        final List<DilemmaMove> opponentMoves = new ArrayList<>();
        final GameHistoryCollection history = player.getGameHistoryCollection();
        final int numGames = history.numberOfPlayedGames();
        for (int i = 0; i < amountOfGames; i++) {
            if (i < numGames) {
                opponentMoves.add(
                        (DilemmaMove) history.getSpecificGameHistory(i).getOpponentMove(0));
            } else {
                opponentMoves.add(null);
            }
        }
        if (opponentMoves.isEmpty()) {
            return Optional.of(initialMove);
        }
        Integer countBeSilentMoves = 0;
        for (final DilemmaMove dilemmaMove : opponentMoves) {
            if (dilemmaMove instanceof DilemmaBeSilentMove) {
                countBeSilentMoves++;
            }
        }

        return getMoveByPercentage(Double.valueOf(countBeSilentMoves), Double.valueOf(opponentMoves.size()));
    }

    /**
     * Returns the move choice of the current player by integrating the number based evaluation.
     * 
     * @param countBeSilentMoves    The amount of {@link DilemmaBeSilentMove}.
     * @param amountOfOppenentMoves The amount of opponent moves which are played.
     * @return
     */
    private Optional<DilemmaMove> getMoveByPercentage(final Double countBeSilentMoves, 
            final Double amountOfOppenentMoves) {
        final Double percent = countBeSilentMoves / amountOfOppenentMoves;
        if (percent == 1.0) {
            return Optional.of(this.providedMoveData.get(IDENTIFIER_100_BESILENT));
        } else if (percent > 0.5) {
            return Optional.of(this.providedMoveData.get(IDENTIFIER_50_BESILENT));
        } else if (percent <= 0.5 && percent > 0.0) {
            return Optional.of(this.providedMoveData.get(IDENTIFIER_50_CONFESS));
        } else {
            return Optional.of(this.providedMoveData.get(IDENTIFIER_100_CONFESS));
        }
    }
}
