/*
 * Copyright © 2021 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel21-freizeit.
 *
 * Ipspiel21-freizeit is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public
 * License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * Ipspiel21-freizeit is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied  * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with ipspiel21-freizeit. If not, see
 * <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel21.freizeit.domain.impl;

import java.util.Map;
import java.util.Optional;

import de.fhdw.gaming.core.domain.DefaultGame;
import de.fhdw.gaming.core.domain.ObserverFactoryProvider;
import de.fhdw.gaming.ipspiel21.freizeit.domain.FreizeitGame;
import de.fhdw.gaming.ipspiel21.freizeit.domain.FreizeitMoveChecker;
import de.fhdw.gaming.ipspiel21.freizeit.domain.FreizeitPlayer;
import de.fhdw.gaming.ipspiel21.freizeit.domain.FreizeitState;
import de.fhdw.gaming.ipspiel21.freizeit.domain.FreizeitStrategy;
import de.fhdw.gaming.ipspiel21.freizeit.moves.FreizeitMove;
import de.fhdw.gaming.ipspiel21.freizeit.moves.factory.FreizeitMoveFactory;
import de.fhdw.gaming.ipspiel21.freizeit.moves.impl.FreizeitDefaultMoveFactory;

/**
 * Implements the Freizeit game.
 */
final class FreizeitGameImpl extends DefaultGame<FreizeitPlayer, FreizeitState, FreizeitMove, FreizeitStrategy>
        implements FreizeitGame {

    /**
     * The move factory.
     */
    private final FreizeitMoveFactory moveFactory;

    /**
     * Creates a Freizeit game.
     *
     * @param id                        The ID of this game.
     * @param initialState              The initial state of the game.
     * @param strategies                The players' strategies.
     * @param maxComputationTimePerMove The maximum computation time per move in seconds.
     * @param moveChecker               The move checker.
     * @param observerFactoryProvider   The {@link ObserverFactoryProvider}.
     * @throws IllegalArgumentException if the player sets do not match.
     * @throws InterruptedException     if creating the game has been interrupted.
     */
    FreizeitGameImpl(final int id, final FreizeitState initialState, final Map<String, FreizeitStrategy> strategies,
            final long maxComputationTimePerMove, final FreizeitMoveChecker moveChecker,
            final ObserverFactoryProvider observerFactoryProvider)
            throws IllegalArgumentException, InterruptedException {

        super(id, initialState, strategies, maxComputationTimePerMove, moveChecker, observerFactoryProvider);
        this.moveFactory = new FreizeitDefaultMoveFactory();
    }

    @Override
    public Optional<FreizeitMove> chooseRandomMove(final FreizeitPlayer player, final FreizeitState state) {
        // choose "Cinema" to punish lame strategies
        return Optional.of(this.moveFactory.createCinemaMove());
    }

    @Override
    public String toString() {
        return String.format("FreizeitGame[id=%d, %s]", this.getId(), this.gameToString());
    }
}
