/*
 * Copyright © 2021 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel21-kopfzahl.
 *
 * Ipspiel21-kopfzahl is free software: you can redistribute it and/or modify it under 
 * the terms of the GNU General Public
 * License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * Ipspiel21-kopfzahl is distributed in the hope that it will be useful, but WITHOUT 
 * ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with ipspiel21-kopfzahl. If not, see
 * <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel21.kopfzahl.domain.impl;

import java.util.LinkedHashMap;
import java.util.LinkedHashSet;

import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.PlayerState;
import de.fhdw.gaming.ipspiel21.kopfzahl.domain.KopfzahlPlayer;
import de.fhdw.gaming.ipspiel21.kopfzahl.domain.KopfzahlPlayerBuilder;
import de.fhdw.gaming.ipspiel21.kopfzahl.domain.KopfzahlState;
import de.fhdw.gaming.ipspiel21.kopfzahl.domain.SaidMove;

/**
 * Implements {@link KopfzahlState}.
 */

final class KopfzahlStateImpl implements KopfzahlState {
    /**
     * Store the moves/answer of players.
     */
    private final Map<KopfzahlPlayer, SaidMove> playerMoves;

    /**
     * Store the outcomes of players.
     */
    private final Map<String, Double> playerOutcomes;
    /**
     * Store the states of players.
     */
    private final Map<String, PlayerState> playerStates;
    /**
     * The first player.
     */
    private final KopfzahlPlayer firstPlayer;
    /**
     * The second player.
     */
    private final KopfzahlPlayer secondPlayer;

    /**
     * Creates a Kopfzahl state.
     *
     * @param firstPlayerBuilder  A builder for the first player.
     * @param secondPlayerBuilder A builder for the second player.
     * @throws GameException if the state cannot be created according to the rules
     *                       of the game.
     */
    KopfzahlStateImpl(final KopfzahlPlayerBuilder firstPlayerBuilder, final KopfzahlPlayerBuilder secondPlayerBuilder)
            throws GameException {

        this.firstPlayer = Objects.requireNonNull(firstPlayerBuilder, "firstPlayerBuilder").build(this);
        this.secondPlayer = Objects.requireNonNull(secondPlayerBuilder, "secondPlayerBuilder").build(this);

        this.playerOutcomes = new LinkedHashMap<>();

        this.playerStates = new LinkedHashMap<>();
        this.playerMoves = new LinkedHashMap<>();
        this.playerStates.put(this.firstPlayer.getName(), PlayerState.PLAYING);
        this.playerStates.put(this.secondPlayer.getName(), PlayerState.PLAYING);

        if (this.firstPlayer.getName().equals(this.secondPlayer.getName())) {
            throw new IllegalArgumentException(
                    String.format("Both players have the same name '%s'.", this.firstPlayer.getName()));
        }
    }

    /**
     * Creates a Kopfzahl state by copying an existing one.
     *
     * @param source The state to copy.
     */
    KopfzahlStateImpl(final KopfzahlStateImpl source) {
        this.firstPlayer = source.firstPlayer.deepCopy(this);
        this.secondPlayer = source.secondPlayer.deepCopy(this);
        this.playerOutcomes = new LinkedHashMap<>(source.playerOutcomes);

        this.playerStates = new LinkedHashMap<>();
        this.playerMoves = new LinkedHashMap<>();
    }

    /**
     * Returns the first player.
     */
    @Override
    public KopfzahlPlayer getFirstPlayer() {
        return this.firstPlayer;
    }

    /**
     * Returns the second player.
     */
    @Override
    public KopfzahlPlayer getSecondPlayer() {
        return this.secondPlayer;
    }

    @Override
    public String toString() {
        return String.format("KopfzahlState[firstPlayer=%s, secondPlayer=%s]", this.firstPlayer, this.secondPlayer);
    }

    @Override
    public boolean equals(final Object obj) {
        if (obj instanceof KopfzahlStateImpl) {
            final KopfzahlStateImpl other = (KopfzahlStateImpl) obj;
            return this.firstPlayer.equals(other.firstPlayer) && this.secondPlayer.equals(other.secondPlayer);
        }
        return false;
    }

    @Override
    public KopfzahlState deepCopy() {
        return this;
    }

    @Override
    public int hashCode() {
        return Objects.hash(this.firstPlayer, this.secondPlayer);
    }

    @Override
    public Map<String, KopfzahlPlayer> getPlayers() {
        final Map<String, KopfzahlPlayer> result = new LinkedHashMap<>();
        result.put(this.firstPlayer.getName(), this.firstPlayer);
        result.put(this.secondPlayer.getName(), this.secondPlayer);
        return result;
    }

    @Override
    public PlayerState getPlayerState(final String playerName) throws IllegalArgumentException {
        final PlayerState playerState = this.playerStates.get(playerName);
        if (playerState != null) {
            return playerState;
        } else {
            throw new IllegalArgumentException(String.format("Unknown player %s.", playerName));
        }

    }
    
    @Override
    public void setPlayerState(final String playerName, final PlayerState newState) throws IllegalArgumentException {
        if (this.playerStates.containsKey(playerName)) {
            this.playerStates.put(playerName, newState);
            if (newState.equals(PlayerState.PLAYING)) {
                this.playerOutcomes.remove(playerName);
            }
        } else {
            throw new IllegalArgumentException(String.format("Unknown player %s.", playerName));
        }
    }

    @Override
    public Optional<Double> getPlayerOutcome(final String playerName) throws IllegalArgumentException {
        if (this.playerStates.containsKey(playerName)) {
            final Double outcome = this.playerOutcomes.get(playerName);
            return outcome != null ? Optional.of(outcome) : KopfzahlState.super.getPlayerOutcome(playerName);
        } else {
            throw new IllegalArgumentException(String.format("Unknown player %s.", playerName));
        }
        // NOTE: use implementation in KopfzahlState when no outcome for the player has
        // been set yet

    }
   

    @Override
    public void setPlayerOutcome(final String playerName, final double newOutcome) throws IllegalArgumentException {
        if (this.playerStates.containsKey(playerName)) {
            this.playerOutcomes.put(playerName, newOutcome);
        } else {
            throw new IllegalArgumentException(String.format("Unknown player %s was found.", playerName));
        } 
    }

    /**
     * get the answer of player.
     * 
     * @param player
     * @return move of the player
     */
    public SaidMove getPlayerMove(final KopfzahlPlayer player) {
        return this.playerMoves.get(player);

    }

    // store answer of player and reject unknown players or players that have
    // already given an answer
    @Override
    public void setAnswer(final KopfzahlPlayer player, final SaidMove answer) {
        try {
            if (!this.playerMoves.containsKey(player)) {
                this.playerMoves.put(player, answer);
            } else {
                throw new IllegalArgumentException(String.format("Player: %s have already given an answer.", 
                        player.getName()));
            }
        } catch (Exception e) {
            throw new IllegalArgumentException(e);
        }
    }

    // add players to set which have not given an answer yet, gibt dir eine Set m
    @Override
    public Set<KopfzahlPlayer> computeNextPlayers() {
        final Set<KopfzahlPlayer> playersWithoutAnswer = new LinkedHashSet<>();
        if (!this.playerMoves.containsKey(this.getFirstPlayer())) {
            playersWithoutAnswer.add(firstPlayer);
        }
        if (!this.playerMoves.containsKey(this.getSecondPlayer())) {
            playersWithoutAnswer.add(secondPlayer);
        }
        return playersWithoutAnswer;
    }

    // compute outcomes and states per player and set them accordingly
    @Override
    public void nextTurn() {

        if (this.computeNextPlayers().isEmpty()) {

            generateOutcomes(this.firstPlayer, this.secondPlayer);
            generateOutcomes(this.secondPlayer, this.firstPlayer);

            generatePlayerState(firstPlayer);
            generatePlayerState(secondPlayer);
        }
    }

    /**
     * compute outcomes per player and set them accordingly.
     * 
     * @param playerOne
     * @param playerTwo
     * @return
     */
    private void generateOutcomes(final KopfzahlPlayer playerOne, final KopfzahlPlayer playerTwo) {
        final SaidMove saidState1 = getPlayerMove(playerOne);
        final SaidMove saidState2 = getPlayerMove(playerTwo);
        final Double outcome = playerOne.getPossibleOutcomes().get(saidState1).get(saidState2);

        setPlayerOutcome(playerOne.getName(), outcome);

    }

    /**
     * compute states per player and set them accordingly.
     * 
     * @param player
     */
    private void generatePlayerState(final KopfzahlPlayer player) {
        if (player.getOutcome().get() > 0) {
            this.setPlayerState(player.getName(), PlayerState.WON);
        } else {
            this.setPlayerState(player.getName(), PlayerState.LOST);
        }
    }
}
