/*
 * Copyright © 2021 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel21-kopfzahlkante.
 *
 * Ipspiel21-kopfzahlkante is free software: you can redistribute it and/or modify it under 
 * the terms of the GNU General Public
 * License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * Ipspiel21-kopfzahlkante is distributed in the hope that it will be useful, but WITHOUT ANY
 *  WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with ipspiel21-kopfzahlkante. If not, see
 * <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel21.kopfzahlkante.domain.impl;

import java.util.Collection;
import java.util.Iterator;

import java.util.Map;
import java.util.Optional;

import de.fhdw.gaming.core.domain.GameException;

import de.fhdw.gaming.ipspiel21.kopfzahlkante.domain.KopfzahlkantePlayer;
import de.fhdw.gaming.ipspiel21.kopfzahlkante.domain.KopfzahlkantePlayerBuilder;
import de.fhdw.gaming.ipspiel21.kopfzahlkante.domain.KopfzahlkanteState;
import de.fhdw.gaming.ipspiel21.kopfzahlkante.domain.SaidMove;

/**
 * Implements {@link KopfzahlkantePlayerBuilder}.
 */
final class KopfzahlkantePlayerBuilderImpl implements KopfzahlkantePlayerBuilder {

    /**
     * The name of the player.
     */
    private Optional<String> name;
    /**
     * The possible outcomes of this player. The key for the first-level map is the
     * answer of the first player, the key for the second-level map is the answer of
     * the second player.
     */
    private Optional<Map<SaidMove, Map<SaidMove, Double>>> possibleOutcomes;

    /**
     * Creates an {@link KopfzahlkantePlayerBuilderImpl}.
     */
    KopfzahlkantePlayerBuilderImpl() {
        this.name = Optional.empty();
        this.possibleOutcomes = Optional.empty();
    }

    @Override
    public KopfzahlkantePlayerBuilderImpl changeName(final String newName) {
        this.name = Optional.of(newName);
        return this;
    }

    @Override
    public KopfzahlkantePlayerBuilder changePossibleOutcomes(
            final Map<SaidMove, Map<SaidMove, Double>> newPossibleOutcomes) {
        this.possibleOutcomes = Optional.of(newPossibleOutcomes);
        return this;
    }

    @Override
    public KopfzahlkantePlayer build(final KopfzahlkanteState state) throws GameException {
        return new KopfzahlkantePlayerImpl(state, this.name.orElseThrow(),
                this.checkPossibleOutcomes(this.possibleOutcomes.orElseThrow()));
    }

    /**
     * Checks if all possible outcomes are defined for a player.
     *
     * @param outcomes The possible outcomes for the player.
     */
    private Map<SaidMove, Map<SaidMove, Double>> checkPossibleOutcomes(
            final Map<SaidMove, Map<SaidMove, Double>> outcomes) {
        if (!isOutcomeNullOrEmptyOutterMap(outcomes) && !isOutcomeNullOrEmptyInnerMap(outcomes.values())) {
            return outcomes;
        } else {
            throw new IllegalArgumentException();
        }
    }

    /**
     * checks if the outer Map of possibleOutcomes is empty or if there is a key
     * with a Null-Value.
     * 
     * @param outcomes
     * @return false if conditions are false else Exceptions
     */
    private static boolean isOutcomeNullOrEmptyOutterMap(final Map<SaidMove, Map<SaidMove, Double>> outcomes) {

        if (outcomes.isEmpty()) {
            throw new IllegalArgumentException("Possible outcomes are not defined for the player");
        } else {
            final Iterator<Map<SaidMove, Double>> it = outcomes.values().iterator();
            while (it.hasNext()) {
                if (it.next() == null) {
                    throw new IllegalArgumentException("The input is not defined:" + it.next());
                }
            }
            return false;
        }
    }

    /**
     * checks if the inner Map of possibleOutcomes is empty or if there is a key
     * with a Null-Value.
     * 
     * @param outcomes
     * @return false if conditions are false else Exceptions
     */
    private static boolean isOutcomeNullOrEmptyInnerMap(final Collection<Map<SaidMove, Double>> outcomes) {
        if (outcomes.isEmpty()) {
            throw new IllegalArgumentException("Possible outcomes are not defined for the player");
        } else {
            final Iterator<Map<SaidMove, Double>> it = outcomes.iterator();
            while (it.hasNext()) {
                if (it.next().values() == null) {
                    throw new IllegalArgumentException("The value is not defined: " + it.next());
                }
            }
            return false;
        }
    }

}
