/*
 * Copyright © 2021 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel21-tictactoe-gui.
 *
 * ipspiel21-tictactoe-gui is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * ipspiel21-tictactoe-gui is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * ipspiel21-tictactoe-gui. If not, see <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel21.tictactoe.gui.impl;

import de.fhdw.gaming.ipspiel21.tictactoe.core.domain.TicTacToeFieldState;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.beans.value.ObservableValue;
import javafx.scene.canvas.Canvas;
import javafx.scene.canvas.GraphicsContext;
import javafx.scene.layout.Region;
import javafx.scene.paint.Color;
import javafx.util.Duration;

/**
 * Displays a Tic Tac Toe field.
 */
final class TicTacToeFieldView extends Region {

    /**
     * The field state.
     */
    private TicTacToeFieldState fieldState;
    /**
     * The canvas.
     */
    private final Canvas canvas;
    /**
     * The duration of a move.
     */
    private final ObjectProperty<Duration> moveDuration;
    /**
     * {@code true} if the field is highlighted, else {@code false}.
     */
    private boolean highlighted;

    /**
     * Creates an {@link TicTacToeFieldView}.
     *
     * @param fieldState The field state.
     */
    TicTacToeFieldView(final TicTacToeFieldState fieldState) {
        this.fieldState = fieldState;
        this.moveDuration = new SimpleObjectProperty<>(Duration.millis(500.0));
        this.highlighted = false;

        this.canvas = new Canvas() {

            @Override
            public boolean isResizable() {
                return true;
            }

            @Override
            public double prefWidth(final double height) {
                return 0.0;
            }

            @Override
            public double prefHeight(final double width) {
                return 0.0;
            }

            @Override
            public double maxWidth(final double height) {
                return Double.POSITIVE_INFINITY;
            }

            @Override
            public double maxHeight(final double width) {
                return Double.POSITIVE_INFINITY;
            }
        };

        this.canvas.widthProperty().bind(this.widthProperty());
        this.canvas.heightProperty().bind(this.heightProperty());
        this.getChildren().add(this.canvas);

        this.widthProperty().addListener(
                (final ObservableValue<? extends Number> observable, final Number oldValue, final Number newValue) -> {
                    this.draw();
                });
        this.heightProperty().addListener(
                (final ObservableValue<? extends Number> observable, final Number oldValue, final Number newValue) -> {
                    this.draw();
                });

        this.setMinSize(50.0, 50.0);
        this.setMaxSize(Double.POSITIVE_INFINITY, Double.POSITIVE_INFINITY);
    }

    /**
     * Returns the move duration property.
     */
    ObjectProperty<Duration> moveDurationProperty() {
        return this.moveDuration;
    }

    /**
     * Updates the state of the field control.
     *
     * @param fieldState The field state.
     */
    void setFieldState(final TicTacToeFieldState fieldState) {
        this.fieldState = fieldState;
        this.draw();
    }

    /**
     * Updates the highlight state of the field control.
     *
     * @param highlighted {@code true} if the field is highlighted, else {@code false}.
     */
    void setHighlighted(final boolean highlighted) {
        this.highlighted = highlighted;
        this.draw();
    }

    /**
     * Draws the field.
     */
    private void draw() {
        final double size = this.getWidth();

        final GraphicsContext gc = this.canvas.getGraphicsContext2D();
        gc.setFill(this.highlighted ? Color.GREEN : Color.DARKOLIVEGREEN);
        gc.fillRect(0.0, 0.0, size, size);

        gc.setFill(Color.BLACK);
        gc.beginPath();
        gc.moveTo(0.0, 0.0);
        gc.lineTo(size, 0.0);
        gc.lineTo(size, size);
        gc.lineTo(0.0, size);
        gc.lineTo(0.0, 0.0);
        gc.closePath();
        gc.stroke();

        final double margin = size * 0.1;

        switch (this.fieldState) {
        case EMPTY:
            // nothing to do
            break;
        case CROSS:
            gc.setFill(Color.BLACK);
            gc.beginPath();
            gc.moveTo(margin, margin);
            gc.lineTo(size - margin, size - margin);
            gc.moveTo(margin, size - margin);
            gc.lineTo(size - margin, margin);
            gc.closePath();
            gc.stroke();
            break;
        case NOUGHT:
            gc.setFill(Color.WHITE);
            gc.strokeOval(margin, margin, size - 2 * margin, size - 2 * margin);
            break;
        default:
            throw new UnsupportedOperationException(
                    String.format("Unknown Tic Tac Toe field state '%s'.", this.fieldState));
        }
    }
}
