/*
 * Copyright © 2021 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel21-tictactoe-gui.
 *
 * ipspiel21-tictactoe-gui is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * ipspiel21-tictactoe-gui is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * ipspiel21-tictactoe-gui. If not, see <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel21.tictactoe.gui.impl;

import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;

import de.fhdw.gaming.ipspiel21.tictactoe.core.domain.TicTacToeBoard;
import de.fhdw.gaming.ipspiel21.tictactoe.core.domain.TicTacToeField;
import de.fhdw.gaming.ipspiel21.tictactoe.core.domain.TicTacToeFieldState;
import de.fhdw.gaming.ipspiel21.tictactoe.core.domain.TicTacToePlayer;
import de.fhdw.gaming.ipspiel21.tictactoe.core.domain.TicTacToePosition;
import de.fhdw.gaming.ipspiel21.tictactoe.core.domain.TicTacToeState;
import de.fhdw.gaming.ipspiel21.tictactoe.gui.TicTacToeBoardEventProvider;
import de.fhdw.gaming.ipspiel21.tictactoe.gui.event.TicTacToeBoardEvent;
import de.fhdw.gaming.ipspiel21.tictactoe.gui.event.TicTacToeMakeMoveBoardEvent;
import javafx.application.Platform;
import javafx.scene.Cursor;
import javafx.scene.input.MouseButton;
import javafx.scene.input.MouseEvent;

/**
 * Implements user interaction with a TicTacToe board.
 */
final class TicTacToeBoardEventProviderImpl implements TicTacToeBoardEventProvider {

    /**
     * The associated {@link TicTacToeBoardView}.
     */
    private final TicTacToeBoardView boardView;

    /**
     * Constructor.
     *
     * @param boardView The associated {@link TicTacToeBoardView}.
     */
    TicTacToeBoardEventProviderImpl(final TicTacToeBoardView boardView) {
        this.boardView = boardView;
    }

    @Override
    public TicTacToeBoardEvent waitForEvent(final TicTacToePlayer player, final TicTacToeState state) {
        final Map<TicTacToePosition, TicTacToeFieldView> emptyFieldViews = new LinkedHashMap<>();
        for (final Map.Entry<TicTacToePosition, ? extends TicTacToeField> entry : state.getBoard()
                .getFieldsBeing(TicTacToeFieldState.EMPTY).entrySet()) {
            emptyFieldViews.put(entry.getKey(), this.boardView.getFieldView(entry.getKey()).orElseThrow());
        }

        final AtomicReference<TicTacToeBoardEvent> event = new AtomicReference<>();
        final Runnable cleanUp;
        if (emptyFieldViews.isEmpty()) {
            final List<TicTacToeFieldView> fieldViews = this.setupInactiveFields(state.getBoard());
            cleanUp = () -> this.cleanUpFields(fieldViews);
        } else {
            this.setupActiveFields(emptyFieldViews, event);
            cleanUp = () -> this.cleanUpFields(emptyFieldViews.values());
        }

        try {
            this.boardView.getUserInputSemaphore().acquire();
            return event.get();
        } catch (final InterruptedException e) {
            return null;
        } finally {
            Platform.runLater(cleanUp);
        }
    }

    /**
     * Sets up the fields when a move is possible (and hence the set of empty fields is not empty).
     *
     * @param emptyFieldViews The non-empty set of views for active fields.
     * @param event           The event to be set when the user selects an active field.
     */
    private void setupActiveFields(final Map<TicTacToePosition, TicTacToeFieldView> emptyFieldViews,
            final AtomicReference<TicTacToeBoardEvent> event) {
        for (final Map.Entry<TicTacToePosition, TicTacToeFieldView> entry : emptyFieldViews.entrySet()) {
            final TicTacToePosition position = entry.getKey();
            final TicTacToeFieldView fieldView = entry.getValue();
            Platform.runLater(() -> {
                fieldView.setCursor(Cursor.CROSSHAIR);
                fieldView.setHighlighted(true);
                fieldView.setOnMouseClicked((final MouseEvent mouseEvent) -> {
                    if (mouseEvent.getButton() == MouseButton.PRIMARY) {
                        event.set(new TicTacToeMakeMoveBoardEvent(position));
                        this.boardView.getUserInputSemaphore().release();
                    }
                });
            });
        }
    }

    /**
     * Sets up the fields when no move is possible (and hence the set of empty fields is empty).
     *
     * @param board The Tic Tac Toe board.
     * @return A list of all field views on the board.
     */
    private List<TicTacToeFieldView> setupInactiveFields(final TicTacToeBoard board) {
        final List<TicTacToeField> fields = new ArrayList<>();
        board.getFields().forEach((final List<? extends TicTacToeField> list) -> fields.addAll(list));

        final List<TicTacToeFieldView> fieldViews = new ArrayList<>();
        for (final TicTacToeField field : fields) {
            final TicTacToeFieldView fieldView = this.boardView.getFieldView(field.getPosition()).orElseThrow();
            fieldViews.add(fieldView);
            Platform.runLater(() -> {
                fieldView.setCursor(Cursor.CLOSED_HAND);
                fieldView.setOnMouseClicked(null);
            });
        }

        return fieldViews;
    }

    /**
     * Cleans up after user interaction by resetting mouse cursors and removing event handlers.
     *
     * @param fieldViews The modified field views.
     */
    private void cleanUpFields(final Collection<? extends TicTacToeFieldView> fieldViews) {
        for (final TicTacToeFieldView fieldView : fieldViews) {
            fieldView.setCursor(Cursor.DEFAULT);
            fieldView.setHighlighted(false);
            fieldView.setOnMouseClicked(null);
        }
    }

    @Override
    public void cancelWaiting() {
        this.boardView.getUserInputSemaphore().release();
    }
}
