
package de.fhdw.gaming.ipspiel21.viergewinnt.gui.impl;

import java.lang.ref.Reference;
import java.lang.ref.WeakReference;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicReference;

import de.fhdw.gaming.core.domain.Game;
import de.fhdw.gaming.gui.GuiObserver;
import de.fhdw.gaming.ipspiel21.viergewinnt.core.domain.VierGewinntGame;
import de.fhdw.gaming.ipspiel21.viergewinnt.gui.VierGewinntBoardEventProvider;
import javafx.scene.Node;

/**
 * Implements {@link GuiObserver}.
 */
final class VierGewinntGuiObserverImpl implements GuiObserver {

    /**
     * The primary (i.e. first created) {@link VierGewinntGuiObserverImpl} instance of this class.
     */
    private static final AtomicReference<WeakReference<VierGewinntGuiObserverImpl>> INSTANCE = new AtomicReference<>();

    /**
     * The {@link VierGewinntBoardView} objects per Vier Gewinnt game ID.
     */
    private final Map<Integer, VierGewinntBoardView> boardViews;

    /**
     * Creates an {@link VierGewinntGuiObserverImpl}.
     */
    VierGewinntGuiObserverImpl() {
        VierGewinntGuiObserverImpl.INSTANCE.compareAndSet(null, new WeakReference<>(this));
        this.boardViews = new LinkedHashMap<>();
    }

    /**
     * Returns a {@link VierGewinntBoardEventProvider} for a given game.
     *
     * @param gameId The game ID.
     * @return The {@link VierGewinntBoardEventProvider}.
     */
    static Optional<VierGewinntBoardEventProvider> getEventProvider(final int gameId) {
        final VierGewinntGuiObserverImpl instance = Optional.ofNullable(VierGewinntGuiObserverImpl.INSTANCE.get())
                .map(Reference::get).orElse(null);
        if (instance == null) {
            return Optional.empty();
        } else {
            final VierGewinntBoardView boardView = instance.boardViews.get(gameId);
            return boardView == null ? Optional.empty() : Optional.of(new VierGewinntBoardEventProviderImpl(boardView));
        }
    }

    @Override
    public Optional<Node> gameCreated(final Game<?, ?, ?, ?> game) {
        if (game instanceof VierGewinntGame) {
            final VierGewinntBoardView boardView = new VierGewinntBoardView((VierGewinntGame) game);
            this.boardViews.put(game.getId(), boardView);
            return Optional.of(boardView.getNode());
        } else {
            return Optional.empty();
        }
    }

    @Override
    public void gamePaused(final Game<?, ?, ?, ?> game) {
        if (game instanceof VierGewinntGame) {
            final VierGewinntBoardView boardView = this.boardViews.get(game.getId());
            if (boardView != null) {
                boardView.gamePaused((VierGewinntGame) game);
            }
        }
    }

    @Override
    public void gameResumed(final Game<?, ?, ?, ?> game) {
        if (game instanceof VierGewinntGame) {
            final VierGewinntBoardView boardView = this.boardViews.get(game.getId());
            if (boardView != null) {
                boardView.gameResumed((VierGewinntGame) game);
            }
        }
    }

    @Override
    public void gameDestroyed(final Game<?, ?, ?, ?> game) {
        if (game instanceof VierGewinntGame) {
            final VierGewinntBoardView boardView = this.boardViews.remove(game.getId());
            if (boardView != null) {
                boardView.destroy((VierGewinntGame) game);
            }
        }
    }
}
