/*
 * Copyright © 2021 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel21-kopfzahlkante.
 *
 * Ipspiel21-kopfzahlkante is free software: you can redistribute it and/or modify it under 
 * the terms of the GNU General Public
 * License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * Ipspiel21-kopfzahlkante is distributed in the hope that it will be useful, but WITHOUT 
 * ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with ipspiel21-kopfzahlkante. If not, see
 * <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel21.kopfzahlkante.domain.impl;

import java.util.LinkedHashMap;
import java.util.LinkedHashSet;

import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.PlayerState;
import de.fhdw.gaming.ipspiel21.kopfzahlkante.domain.KopfzahlkantePlayer;
import de.fhdw.gaming.ipspiel21.kopfzahlkante.domain.KopfzahlkantePlayerBuilder;
import de.fhdw.gaming.ipspiel21.kopfzahlkante.domain.KopfzahlkanteState;
import de.fhdw.gaming.ipspiel21.kopfzahlkante.domain.SaidMove;

/**
 * Implements {@link KopfzahlkanteState}.
 */

final class KopfzahlkanteStateImpl implements KopfzahlkanteState {
    /**
     * Store the moves/answer of players.
     */
    private final Map<KopfzahlkantePlayer, SaidMove> playerMoves;

    /**
     * Store the outcomes of players.
     */
    private final Map<String, Double> playerOutcomes;
    /**
     * Store the states of players.
     */
    private final Map<String, PlayerState> playerStates;
    /**
     * The first player.
     */
    private final KopfzahlkantePlayer firstPlayer;
    /**
     * The second player.
     */
    private final KopfzahlkantePlayer secondPlayer;

    /**
     * Creates a Kopfzahlkante state.
     *
     * @param firstPlayerBuilder  A builder for the first player.
     * @param secondPlayerBuilder A builder for the second player.
     * @throws GameException if the state cannot be created according to the rules
     *                       of the game.
     */
    KopfzahlkanteStateImpl(final KopfzahlkantePlayerBuilder firstPlayerBuilder,
            final KopfzahlkantePlayerBuilder secondPlayerBuilder) throws GameException {

        this.firstPlayer = Objects.requireNonNull(firstPlayerBuilder, "firstPlayerBuilder").build(this);
        this.secondPlayer = Objects.requireNonNull(secondPlayerBuilder, "secondPlayerBuilder").build(this);

        this.playerOutcomes = new LinkedHashMap<>();

        this.playerStates = new LinkedHashMap<>();
        this.playerMoves = new LinkedHashMap<>();
        this.playerStates.put(this.firstPlayer.getName(), PlayerState.PLAYING);
        this.playerStates.put(this.secondPlayer.getName(), PlayerState.PLAYING);

        if (this.firstPlayer.getName().equals(this.secondPlayer.getName())) {
            throw new IllegalArgumentException(
                    String.format("Both players have the same name '%s'.", this.firstPlayer.getName()));
        }
    }

    /**
     * Creates a Kopfzahlkante state by copying an existing one.
     *
     * @param source The state to copy.
     */
    KopfzahlkanteStateImpl(final KopfzahlkanteStateImpl source) {
        this.firstPlayer = source.firstPlayer.deepCopy(this);
        this.secondPlayer = source.secondPlayer.deepCopy(this);
        this.playerOutcomes = new LinkedHashMap<>(source.playerOutcomes);

        this.playerStates = new LinkedHashMap<>();
        this.playerMoves = new LinkedHashMap<>();
    }

    /**
     * Returns the first player.
     */
    @Override
    public KopfzahlkantePlayer getFirstPlayer() {
        return this.firstPlayer;
    }

    /**
     * Returns the second player.
     */
    @Override
    public KopfzahlkantePlayer getSecondPlayer() {
        return this.secondPlayer;
    }

    @Override
    public String toString() {
        return String.format("KopfzahlkanteState[firstPlayer=%s, secondPlayer=%s]", this.firstPlayer,
                this.secondPlayer);
    }

    @Override
    public boolean equals(final Object obj) {
        if (obj instanceof KopfzahlkanteStateImpl) {
            final KopfzahlkanteStateImpl other = (KopfzahlkanteStateImpl) obj;
            return this.firstPlayer.equals(other.firstPlayer) && this.secondPlayer.equals(other.secondPlayer);
        }
        return false;
    }

    @Override
    public KopfzahlkanteState deepCopy() {
        return this;
    }

    @Override
    public int hashCode() {
        return Objects.hash(this.firstPlayer, this.secondPlayer);
    }

    @Override
    public Map<String, KopfzahlkantePlayer> getPlayers() {
        final Map<String, KopfzahlkantePlayer> result = new LinkedHashMap<>();
        result.put(this.firstPlayer.getName(), this.firstPlayer);
        result.put(this.secondPlayer.getName(), this.secondPlayer);
        return result;
    }

    @Override
    public PlayerState getPlayerState(final String playerName) throws IllegalArgumentException {
        final PlayerState playerState = this.playerStates.get(playerName);
        if (playerState != null) {
            return playerState;
        } else {
            throw new IllegalArgumentException(String.format("Unknown player %s.", playerName));
        }

    }

    @Override
    public void setPlayerState(final String playerName, final PlayerState newState) throws IllegalArgumentException {
        if (this.playerStates.containsKey(playerName)) {
            this.playerStates.put(playerName, newState);
            if (newState.equals(PlayerState.PLAYING)) {
                this.playerOutcomes.remove(playerName);
            }
        } else {
            throw new IllegalArgumentException(String.format("Unknown player %s.", playerName));
        }
    }

    @Override
    public Optional<Double> getPlayerOutcome(final String playerName) throws IllegalArgumentException {
        if (this.playerStates.containsKey(playerName)) {
            final Double outcome = this.playerOutcomes.get(playerName);
            return outcome != null ? Optional.of(outcome) : KopfzahlkanteState.super.getPlayerOutcome(playerName);
        } else {
            throw new IllegalArgumentException(String.format("Unknown player %s.", playerName));
        }
    }

    @Override
    public void setPlayerOutcome(final String playerName, final double newOutcome) throws IllegalArgumentException {
        if (this.playerStates.containsKey(playerName)) {
            this.playerOutcomes.put(playerName, newOutcome);
        } else {
            throw new IllegalArgumentException(String.format("Unknown player %s was found.", playerName));
        }
    }

    /**
     * get the answer of player.
     * 
     * @param player
     * @return move of the player
     */
    public SaidMove getPlayerMove(final KopfzahlkantePlayer player) {
        return this.playerMoves.get(player);

    }

    @Override
    public void setAnswer(final KopfzahlkantePlayer player, final SaidMove answer) {
        try {
            if (!this.playerMoves.containsKey(player)) {
                this.playerMoves.put(player, answer);
            } else {
                throw new IllegalArgumentException(
                        String.format("Player: %s have already given an answer.", player.getName()));
            }
        } catch (Exception e) {
            throw new IllegalArgumentException(e);
        }
    }

    @Override
    public Set<KopfzahlkantePlayer> computeNextPlayers() {
        final Set<KopfzahlkantePlayer> playersWithoutAnswer = new LinkedHashSet<>();
        if (!this.playerMoves.containsKey(this.getFirstPlayer())) {
            playersWithoutAnswer.add(firstPlayer);
        }
        if (!this.playerMoves.containsKey(this.getSecondPlayer())) {
            playersWithoutAnswer.add(secondPlayer);
        }
        return playersWithoutAnswer;
    }

    @Override
    public void nextTurn() {

        if (this.computeNextPlayers().isEmpty()) {

            generateOutcomes(this.firstPlayer, this.secondPlayer);
            generateOutcomes(this.secondPlayer, this.firstPlayer);

            generatePlayerState(firstPlayer);
            generatePlayerState(secondPlayer);
        }
    }

    /**
     * compute outcomes per player and set them accordingly.
     * 
     * @param playerOne
     * @param playerTwo
     * @return
     */
    private void generateOutcomes(final KopfzahlkantePlayer playerOne, final KopfzahlkantePlayer playerTwo) {
        final SaidMove saidState1 = getPlayerMove(playerOne);
        final SaidMove saidState2 = getPlayerMove(playerTwo);
        final Double outcome = playerOne.getPossibleOutcomes().get(saidState1).get(saidState2);

        setPlayerOutcome(playerOne.getName(), outcome);

    }

    /**
     * compute states per player and set them accordingly.
     * 
     * @param player
     */
    private void generatePlayerState(final KopfzahlkantePlayer player) {
        if (player.getOutcome().get() > 0) {
            this.setPlayerState(player.getName(), PlayerState.WON);
        } else {
            this.setPlayerState(player.getName(), PlayerState.LOST);
        }
    }
}
