/*
 * Copyright © 2021 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel21-kopfzahl.
 *
 * Ipspiel21-kopfzahl is free software: you can redistribute it and/or modify it under 
 * the terms of the GNU General Public
 * License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * Ipspiel21-kopfzahl is distributed in the hope that it will be useful, but WITHOUT ANY
 *  WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with ipspiel21-kopfzahl. If not, see
 * <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel21.kopfzahl.domain.impl;

import java.util.Collection;
import java.util.Iterator;

import java.util.Map;
import java.util.Optional;

import de.fhdw.gaming.core.domain.GameException;

import de.fhdw.gaming.ipspiel21.kopfzahl.domain.KopfzahlPlayer;
import de.fhdw.gaming.ipspiel21.kopfzahl.domain.KopfzahlPlayerBuilder;
import de.fhdw.gaming.ipspiel21.kopfzahl.domain.KopfzahlState;
import de.fhdw.gaming.ipspiel21.kopfzahl.domain.SaidMove;

/**
 * Implements {@link KopfzahlPlayerBuilder}.
 */
final class KopfzahlPlayerBuilderImpl implements KopfzahlPlayerBuilder {

    /**
     * The name of the player.
     */
    private Optional<String> name;
    /**
     * The possible outcomes of this player. The key for the first-level map is the answer of the first player, the key
     * for the second-level map is the answer of the second player.
     */
    private Optional<Map<SaidMove, Map<SaidMove, Double>>> possibleOutcomes;

    /**
     * Creates an {@link KopfzahlPlayerBuilderImpl}.
     */
    KopfzahlPlayerBuilderImpl() {
        this.name = Optional.empty();
        this.possibleOutcomes = Optional.empty();
    }

    @Override
    public KopfzahlPlayerBuilderImpl changeName(final String newName) {
        this.name = Optional.of(newName);
        return this;
    }

    @Override
public KopfzahlPlayerBuilder changePossibleOutcomes(final Map<SaidMove, 
Map<SaidMove, Double>> newPossibleOutcomes) {
        this.possibleOutcomes = Optional.of(newPossibleOutcomes);
        return this;
   }

    @Override
    public KopfzahlPlayer build(final KopfzahlState state) throws GameException {
        return new KopfzahlPlayerImpl(
                state,
                this.name.orElseThrow(),
                this.checkPossibleOutcomes(this.possibleOutcomes.orElseThrow()));
    }
  /**
     * Checks if all possible outcomes are defined for a player.
     *
     * @param outcomes The possible outcomes for the player.
     */
    private Map<SaidMove, Map<SaidMove, Double>> checkPossibleOutcomes(
            final Map<SaidMove, Map<SaidMove, Double>> outcomes) {
       if (!isOutcomeNullOrEmptyOutterMap(outcomes) 
               && !isOutcomeNullOrEmptyInnerMap(outcomes.values())) {
           return outcomes;
       } else { 
           throw new IllegalArgumentException();
       }
       }
    
    
    /**
     * checks if the outter Map of possibleOutcomes is empty or if there is a 
     * key with a Null-Value. 
     * @param outcomes
     * @return false if conditions are false else Exceptions
     */
    private static boolean isOutcomeNullOrEmptyOutterMap(
            final Map<SaidMove, Map<SaidMove, Double>> outcomes) {
       
        if (outcomes.isEmpty()) { 
        throw new IllegalArgumentException("Possible outcomes are not defined for the player");
        } else {
        final Iterator<Map<SaidMove, Double>> it = outcomes.values().iterator(); 
        while (it.hasNext()) {
           if (it.next() == null) {
               throw new IllegalArgumentException("The input is not defined:" + it.next());
           }
        }
        return false;
        }
    }
   /**
    * checks if the inner Map of possibleOutcomes is empty or if there is a 
    * key with a Null-Value. 
    * @param outcomes
    * @return false if conditions are false else Exceptions
    */
   private static boolean isOutcomeNullOrEmptyInnerMap(final Collection<Map<SaidMove, Double>> outcomes) {
   if (outcomes.isEmpty()) {
       throw new IllegalArgumentException("Possible outcomes are not defined for the player");
   } else {
  final Iterator<Map<SaidMove, Double>> it = outcomes.iterator();
   while (it.hasNext()) {
//      if (it.next().values() == null) {
        if (it.next().values().isEmpty() || it.next().values() == null) {
          throw new IllegalArgumentException("The input is not defined:" + it.next());
      }
   }
   return false;
   }
   }
    
}
