/*
 * Copyright © 2021 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel21-kopfzahlkante.
 *
 * Ipspiel21-kopfzahlkante is free software: you can redistribute it 
 * and/or modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * Ipspiel21-kopfzahlkante is distributed in the hope that it will 
 * be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with ipspiel21-kopfzahlkante. If not, see
 * <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel21.kopfzahlkante.domain.impl;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Pattern;

import de.fhdw.gaming.core.domain.GameBuilder;
import de.fhdw.gaming.core.domain.GameBuilderFactory;
import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.Strategy;
import de.fhdw.gaming.core.ui.InputProvider;
import de.fhdw.gaming.core.ui.InputProviderException;
import de.fhdw.gaming.core.ui.type.validator.MaxValueValidator;
import de.fhdw.gaming.core.ui.type.validator.MinValueValidator;
import de.fhdw.gaming.core.ui.type.validator.PatternValidator;
import de.fhdw.gaming.ipspiel21.kopfzahlkante.domain.KopfzahlkanteGameBuilder;
import de.fhdw.gaming.ipspiel21.kopfzahlkante.domain.KopfzahlkanteGameBuilderFactory;
import de.fhdw.gaming.ipspiel21.kopfzahlkante.domain.KopfzahlkantePlayerBuilder;
import de.fhdw.gaming.ipspiel21.kopfzahlkante.domain.KopfzahlkanteStrategy;
import de.fhdw.gaming.ipspiel21.kopfzahlkante.domain.SaidMove;
import de.fhdw.gaming.ipspiel21.kopfzahlkante.domain.factory.KopfzahlkanteDefaultStrategyFactoryProvider;
import de.fhdw.gaming.ipspiel21.kopfzahlkante.domain.factory.KopfzahlkanteStrategyFactory;
import de.fhdw.gaming.ipspiel21.kopfzahlkante.domain.factory.KopfzahlkanteStrategyFactoryProvider;
import de.fhdw.gaming.ipspiel21.kopfzahlkante.moves.factory.KopfzahlkanteMoveFactory;
import de.fhdw.gaming.ipspiel21.kopfzahlkante.moves.impl.KopfzahlkanteDefaultMoveFactory;

/**
 * Implements {@link GameBuilderFactory} by creating a Kopfzahlkante game builder.
 */
public final class KopfzahlkanteGameBuilderFactoryImpl implements KopfzahlkanteGameBuilderFactory {

    /**
     * The number of players.
     */
    private static final int NUMBER_OF_PLAYERS = 2;
    /**
     * Smallest allowed maximum computation time per move in seconds.
     */
    private static final int MIN_MAX_COMPUTATION_TIME_PER_MOVE = 1;
    /**
     * Largest allowed maximum computation time per move in seconds.
     */
    private static final int MAX_MAX_COMPUTATION_TIME_PER_MOVE = 3600;

    /**
     * All available Kopfzahlkante strategies.
     */
    private final Set<KopfzahlkanteStrategy> strategies;

    /**
     * Creates a Kopfzahlkante game factory. Kopfzahlkante strategies are loaded by using the
     * {@link java.util.ServiceLoader}.
     * <p>
     * This constructor is meant to be used by the {@link java.util.ServiceLoader}.
     */
    public KopfzahlkanteGameBuilderFactoryImpl() {
        this(new KopfzahlkanteDefaultStrategyFactoryProvider());
    }

    /**
     * Creates a Kopfzahlkante game factory.
     *
     * @param strategyFactoryProvider The {@link KopfzahlkanteStrategyFactoryProvider} for loading Kopfzahlkante
     *                                strategies.
     */
    KopfzahlkanteGameBuilderFactoryImpl(final KopfzahlkanteStrategyFactoryProvider strategyFactoryProvider) {
        final KopfzahlkanteMoveFactory moveFactory = new KopfzahlkanteDefaultMoveFactory();

        final List<KopfzahlkanteStrategyFactory> factories = strategyFactoryProvider.getStrategyFactories();
        this.strategies = new LinkedHashSet<>();
        for (final KopfzahlkanteStrategyFactory factory : factories) {
            this.strategies.add(factory.create(moveFactory));
        }
    }

    @Override
    public String getName() {
        return "Kopfzahlkante";
    }

    @Override
    public int getMinimumNumberOfPlayers() {
        return KopfzahlkanteGameBuilderFactoryImpl.NUMBER_OF_PLAYERS;
    }

    @Override
    public int getMaximumNumberOfPlayers() {
        return KopfzahlkanteGameBuilderFactoryImpl.NUMBER_OF_PLAYERS;
    }

    @Override
    public List<? extends Strategy<?, ?, ?>> getStrategies() {
        return new ArrayList<>(this.strategies);
    }

    @Override
    public KopfzahlkanteGameBuilder createGameBuilder(final InputProvider inputProvider) throws GameException {
        try {
            final KopfzahlkanteGameBuilder gameBuilder = new KopfzahlkanteGameBuilderImpl();

            @SuppressWarnings("unchecked")
            final Map<String, Object> gameData = inputProvider.needInteger(
                    GameBuilderFactory.PARAM_MAX_COMPUTATION_TIME_PER_MOVE,
                    "Maximum computation time per move in seconds",
                    Optional.of(GameBuilder.DEFAULT_MAX_COMPUTATION_TIME_PER_MOVE),
                    new MinValueValidator<>(KopfzahlkanteGameBuilderFactoryImpl.MIN_MAX_COMPUTATION_TIME_PER_MOVE),
                    new MaxValueValidator<>(KopfzahlkanteGameBuilderFactoryImpl.MAX_MAX_COMPUTATION_TIME_PER_MOVE))
                    .requestData("Game properties");

            gameBuilder.changeMaximumComputationTimePerMove(
                    (Integer) gameData.get(GameBuilderFactory.PARAM_MAX_COMPUTATION_TIME_PER_MOVE));

            final InputProvider firstPlayerInputProvider = inputProvider.getNext(gameData);
            final Map<String, Object> firstPlayerData = this.requestPlayer1Data(firstPlayerInputProvider, "Player 1");
            final KopfzahlkantePlayerBuilder firstPlayerBuilder = this
                    .initPlayerBuilder(gameBuilder.createPlayerBuilder(), firstPlayerData);
            final KopfzahlkanteStrategy firstPlayerStrategy = this.getStrategy(firstPlayerData);
            gameBuilder.addPlayerBuilder(firstPlayerBuilder, firstPlayerStrategy);

            final InputProvider secondPlayerInputProvider = firstPlayerInputProvider.getNext(firstPlayerData);
            final Map<String, Object> secondPlayerData = this.requestPlayer2Data(secondPlayerInputProvider, "Player 2");
            final KopfzahlkantePlayerBuilder secondPlayerBuilder = this
                    .initPlayerBuilder(gameBuilder.createPlayerBuilder(), secondPlayerData);
            final KopfzahlkanteStrategy secondPlayerStrategy = this.getStrategy(secondPlayerData);
            gameBuilder.addPlayerBuilder(secondPlayerBuilder, secondPlayerStrategy);

            return gameBuilder;
        } catch (final InputProviderException e) {
            throw new GameException(String.format("Creating Kopfzahlkante game was aborted: %s", e.getMessage()), e);
        }
    }

    /**
     * Returns data for a player builder.
     *
     * @param inputProvider The input provider.
     * @param title         The title for the UI.
     * @throws InputProviderException if the operation has been aborted prematurely (e.g. if the user cancelled a
     *                                dialog).
     */
    @SuppressWarnings("unchecked")
    private Map<String, Object> requestPlayer1Data(final InputProvider inputProvider, final String title)
            throws GameException, InputProviderException {

        inputProvider
                .needString(GameBuilderFactory.PARAM_PLAYER_NAME, "Name", Optional.empty(),
                        new PatternValidator(Pattern.compile("\\S+(\\s+\\S+)*")))
                .needInteger(KopfzahlkanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_KOPF_KOPF,
                        "Player's outcome on Kopf/Kopf", Optional.of(1))
                .needInteger(KopfzahlkanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_KOPF_ZAHL,
                        "Player's outcome on Kopf/Zahl", Optional.of(-1))
                .needInteger(KopfzahlkanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_KOPF_KANTE,
                        "Player's outcome on Kopf/Kante", Optional.of(-1))
                .needInteger(KopfzahlkanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_ZAHL_KOPF,
                        "Player's outcome on Zahl/Kopf", Optional.of(-1))
                .needInteger(KopfzahlkanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_ZAHL_ZAHL,
                        "Player's outcome on Zahl/Zahl", Optional.of(1))
                .needInteger(KopfzahlkanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_ZAHL_KANTE,
                        "Player's outcome on Zahl/Kante", Optional.of(-1))
                .needInteger(KopfzahlkanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_KANTE_KOPF,
                        "Player's outcome on Kante/Kopf", Optional.of(-1))
                .needInteger(KopfzahlkanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_KANTE_ZAHL,
                        "Player's outcome on Kante/Zahl", Optional.of(-1))
                .needInteger(KopfzahlkanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_KANTE_KANTE,
                        "Player's outcome on Kante/Kante", Optional.of(1))
                .needObject(GameBuilderFactory.PARAM_PLAYER_STRATEGY, "Strategy", Optional.empty(), this.strategies);

        return inputProvider.requestData(title);
    }

    /**
     * Returns data for a player builder.
     *
     * @param inputProvider The input provider.
     * @param title         The title for the UI.
     * @throws InputProviderException if the operation has been aborted prematurely (e.g. if the user cancelled a
     *                                dialog).
     */
    @SuppressWarnings("unchecked")
    private Map<String, Object> requestPlayer2Data(final InputProvider inputProvider, final String title)
            throws GameException, InputProviderException {

        inputProvider
                .needString(GameBuilderFactory.PARAM_PLAYER_NAME, "Name", Optional.empty(),
                        new PatternValidator(Pattern.compile("\\S+(\\s+\\S+)*")))
                .needInteger(KopfzahlkanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_KOPF_KOPF,
                        "Player's outcome on Kopf/Kopf", Optional.of(-1))
                .needInteger(KopfzahlkanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_KOPF_ZAHL,
                        "Player's outcome on Kopf/Zahl", Optional.of(1))
                .needInteger(KopfzahlkanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_KOPF_KANTE,
                        "Player's outcome on Kopf/Kante", Optional.of(1))
                .needInteger(KopfzahlkanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_ZAHL_KOPF,
                        "Player's outcome on Zahl/Kopf", Optional.of(1))
                .needInteger(KopfzahlkanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_ZAHL_ZAHL,
                        "Player's outcome on Zahl/Zahl", Optional.of(-1))
                .needInteger(KopfzahlkanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_ZAHL_KANTE,
                        "Player's outcome on Zahl/Kante", Optional.of(1))
                .needInteger(KopfzahlkanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_KANTE_KOPF,
                        "Player's outcome on Kante/Kopf", Optional.of(1))
                .needInteger(KopfzahlkanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_KANTE_ZAHL,
                        "Player's outcome on Kante/Zahl", Optional.of(1))
                .needInteger(KopfzahlkanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_KANTE_KANTE,
                        "Player's outcome on Kante/Kante", Optional.of(-1))
                .needObject(GameBuilderFactory.PARAM_PLAYER_STRATEGY, "Strategy", Optional.empty(), this.strategies);

        return inputProvider.requestData(title);
    }

    /**
     * Initialises a Kopfzahlkante player builder.
     *
     * @param playerBuilder The player builder.
     * @param playerData    The requested player data.
     * @return {@code playerBuilder}.
     * @throws InputProviderException if the operation has been aborted prematurely (e.g. if the user cancelled a
     *                                dialog).
     */
    private KopfzahlkantePlayerBuilder initPlayerBuilder(final KopfzahlkantePlayerBuilder playerBuilder,
            final Map<String, Object> playerData) throws GameException, InputProviderException {

        final Map<SaidMove, Map<SaidMove, Double>> possibleOutcomes = new LinkedHashMap<>();

        final Map<SaidMove, Double> possibleOutcomesKopf = new LinkedHashMap<>();
        possibleOutcomesKopf.put(SaidMove.KOPF,
                (double) (Integer) playerData.get(KopfzahlkanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_KOPF_KOPF));
        possibleOutcomesKopf.put(SaidMove.ZAHL,
                (double) (Integer) playerData.get(KopfzahlkanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_KOPF_ZAHL));
        possibleOutcomesKopf.put(SaidMove.KANTE,
                (double) (Integer) playerData.get(KopfzahlkanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_KOPF_KANTE));
        possibleOutcomes.put(SaidMove.KOPF, possibleOutcomesKopf);

        final Map<SaidMove, Double> possibleOutcomesZahl = new LinkedHashMap<>();
        possibleOutcomesZahl.put(SaidMove.KOPF,
                (double) (Integer) playerData.get(KopfzahlkanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_ZAHL_KOPF));
        possibleOutcomesZahl.put(SaidMove.ZAHL,
                (double) (Integer) playerData.get(KopfzahlkanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_ZAHL_ZAHL));
        possibleOutcomesZahl.put(SaidMove.KANTE,
                (double) (Integer) playerData.get(KopfzahlkanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_ZAHL_KANTE));
        possibleOutcomes.put(SaidMove.ZAHL, possibleOutcomesZahl);

        final Map<SaidMove, Double> possibleOutcomesKante = new LinkedHashMap<>();
        possibleOutcomesKante.put(SaidMove.KOPF,
                (double) (Integer) playerData.get(KopfzahlkanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_KANTE_KOPF));
        possibleOutcomesKante.put(SaidMove.ZAHL,
                (double) (Integer) playerData.get(KopfzahlkanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_KANTE_ZAHL));
        possibleOutcomesKante.put(SaidMove.KANTE,
                (double) (Integer) playerData.get(KopfzahlkanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_KANTE_KANTE));
        possibleOutcomes.put(SaidMove.KANTE, possibleOutcomesKante);

        return playerBuilder.changeName((String) playerData.get(GameBuilderFactory.PARAM_PLAYER_NAME))
                .changePossibleOutcomes(possibleOutcomes);
    }

    /**
     * Returns a Kopfzahlkante strategy.
     *
     * @param playerData The requested player data.
     * @return The Kopfzahlkante strategy.
     */
    private KopfzahlkanteStrategy getStrategy(final Map<String, Object> playerData) {
        return (KopfzahlkanteStrategy) playerData.get(GameBuilderFactory.PARAM_PLAYER_STRATEGY);
    }

}
