package de.fhdw.gaming.ipspiel22.kopfundzahl.domain.impl;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import de.fhdw.gaming.core.domain.DefaultObserverFactoryProvider;
import de.fhdw.gaming.core.domain.GameBuilder;
import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.ObserverFactoryProvider;
import de.fhdw.gaming.ipspiel22.kopfundzahl.domain.KopfundZahlGame;
import de.fhdw.gaming.ipspiel22.kopfundzahl.domain.KopfundZahlGameBuilder;
import de.fhdw.gaming.ipspiel22.kopfundzahl.domain.KopfundZahlPlayer;
import de.fhdw.gaming.ipspiel22.kopfundzahl.domain.KopfundZahlPlayerBuilder;
import de.fhdw.gaming.ipspiel22.kopfundzahl.domain.KopfundZahlStrategy;
import de.fhdw.gaming.ipspiel22.kopfundzahl.moves.impl.AbstractKopfundZahlMove;

/**
 * Implements {@link KopfundZahlGameBuilder}.
 */
public final class KopfundZahlGameBuilderImpl implements KopfundZahlGameBuilder {

    /**
     * The {@link ObserverFactoryProvider}.
     */
    private ObserverFactoryProvider observerFactoryProvider;

    /**
     * The builder for the firstPlayer.
     */
    private Optional<KopfundZahlPlayer> firstPlayer;

    /**
     * The strategy of the firstplayer.
     */
    private Optional<KopfundZahlStrategy> firstPlayerStrategy;

    /**
     * The builder for the secondPlayer.
     */
    private Optional<KopfundZahlPlayer> secondPlayer;

    /**
     * The strategy of the secondplayer.
     */
    private Optional<KopfundZahlStrategy> secondPlayerStrategy;

    /**
     * The maximum computation time per move in seconds.
     */
    private int maxComputationTimePerMove;

    /**
     * Creates a KopfundZahl game builder.
     */
    public KopfundZahlGameBuilderImpl() {
        this.observerFactoryProvider = new DefaultObserverFactoryProvider();
        this.firstPlayer = Optional.empty();
        this.firstPlayerStrategy = Optional.empty();
        this.secondPlayer = Optional.empty();
        this.secondPlayerStrategy = Optional.empty();
        this.maxComputationTimePerMove = GameBuilder.DEFAULT_MAX_COMPUTATION_TIME_PER_MOVE;
    }

    @Override
    public KopfundZahlPlayerBuilder createPlayerBuilder() {
        return new KopfundZahlPlayerBuilderImpl();
    }

    @Override
    public KopfundZahlGameBuilder addPlayer(final KopfundZahlPlayer player,
            final KopfundZahlStrategy strategy)
            throws GameException {

        if (this.firstPlayer.isEmpty()) {
            this.firstPlayer = Optional.of(Objects.requireNonNull(player, "player"));
            this.firstPlayerStrategy = Optional.of(Objects.requireNonNull(strategy, "firstPlayerStrategy"));
        } else if (this.secondPlayer.isEmpty()) {
            this.secondPlayer = Optional.of(Objects.requireNonNull(player, "player"));
            this.secondPlayerStrategy = Optional.of(Objects.requireNonNull(strategy, "secondPlayerStrategy"));
        } else {
            throw new GameException(String.format("More than two players are now allowed."));
        }
        return this;
    }

    @Override
    public GameBuilder changeMaximumComputationTimePerMove(final int newMaxComputationTimePerMove) {
        this.maxComputationTimePerMove = newMaxComputationTimePerMove;
        return this;
    }

    @Override
    public KopfundZahlGameBuilder changeObserverFactoryProvider(
            final ObserverFactoryProvider newObserverFactoryProvider) {
        this.observerFactoryProvider = newObserverFactoryProvider;
        return this;
    }

    @Override
    public KopfundZahlGame build(final int id) throws GameException, InterruptedException {
        if (!this.firstPlayer.isPresent() || !this.secondPlayer.isPresent()) {
            throw new GameException("A Head and Tail game needs two players.");
        }

        final KopfundZahlStateImpl initialState = new KopfundZahlStateImpl(this.firstPlayer.get(),
                this.secondPlayer.get());

        final Map<String, KopfundZahlStrategy> strategies = new LinkedHashMap<>();
        strategies.put(initialState.getFirstPlayer().getName(), this.firstPlayerStrategy.orElseThrow());
        strategies.put(initialState.getSecondPlayer().getName(), this.secondPlayerStrategy.orElseThrow());
        return new KopfundZahlGameImpl(
                id,
                initialState,
                strategies,
                this.maxComputationTimePerMove,
                AbstractKopfundZahlMove.class::isInstance,
                this.observerFactoryProvider);
    }

}
