package de.fhdw.gaming.ipspiel22.kopfundzahlundkante.domain.impl;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import de.fhdw.gaming.core.domain.DefaultObserverFactoryProvider;
import de.fhdw.gaming.core.domain.GameBuilder;
import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.ObserverFactoryProvider;
import de.fhdw.gaming.ipspiel22.kopfundzahlundkante.domain.KopfundZahlundKanteGame;
import de.fhdw.gaming.ipspiel22.kopfundzahlundkante.domain.KopfundZahlundKanteGameBuilder;
import de.fhdw.gaming.ipspiel22.kopfundzahlundkante.domain.KopfundZahlundKantePlayer;
import de.fhdw.gaming.ipspiel22.kopfundzahlundkante.domain.KopfundZahlundKantePlayerBuilder;
import de.fhdw.gaming.ipspiel22.kopfundzahlundkante.domain.KopfundZahlundKanteStrategy;
import de.fhdw.gaming.ipspiel22.kopfundzahlundkante.moves.impl.AbstractKopfundZahlundKanteMove;

/**
 * Implements {@link KopfundZahlundKanteGameBuilder}.
 */
public final class KopfundZahlundKanteGameBuilderImpl implements KopfundZahlundKanteGameBuilder {

    /**
     * The {@link ObserverFactoryProvider}.
     */
    private ObserverFactoryProvider observerFactoryProvider;

    /**
     * The builder for the firstPlayer.
     */
    private Optional<KopfundZahlundKantePlayer> firstPlayer;

    /**
     * The strategy of the firstplayer.
     */
    private Optional<KopfundZahlundKanteStrategy> firstPlayerStrategy;

    /**
     * The builder for the secondPlayer.
     */
    private Optional<KopfundZahlundKantePlayer> secondPlayer;

    /**
     * The strategy of the secondplayer.
     */
    private Optional<KopfundZahlundKanteStrategy> secondPlayerStrategy;

    /**
     * The maximum computation time per move in seconds.
     */
    private int maxComputationTimePerMove;

    /**
     * Creates a KopfundZahl game builder.
     */
    public KopfundZahlundKanteGameBuilderImpl() {
        this.observerFactoryProvider = new DefaultObserverFactoryProvider();
        this.firstPlayer = Optional.empty();
        this.firstPlayerStrategy = Optional.empty();
        this.secondPlayer = Optional.empty();
        this.secondPlayerStrategy = Optional.empty();
        this.maxComputationTimePerMove = GameBuilder.DEFAULT_MAX_COMPUTATION_TIME_PER_MOVE;
    }

    @Override
    public KopfundZahlundKantePlayerBuilder createPlayerBuilder() {
        return new KopfundZahlundKantePlayerBuilderImpl();
    }

    @Override
    public KopfundZahlundKanteGameBuilder addPlayer(final KopfundZahlundKantePlayer player,
            final KopfundZahlundKanteStrategy strategy) throws GameException {

        if (this.firstPlayer.isEmpty()) {
            this.firstPlayer = Optional.of(Objects.requireNonNull(player, "player"));
            this.firstPlayerStrategy = Optional.of(Objects.requireNonNull(strategy, "firstPlayerStrategy"));
        } else if (this.secondPlayer.isEmpty()) {
            this.secondPlayer = Optional.of(Objects.requireNonNull(player, "player"));
            this.secondPlayerStrategy = Optional.of(Objects.requireNonNull(strategy, "secondPlayerStrategy"));
        } else {
            throw new GameException(String.format("More than two players are now allowed."));
        }
        return this;
    }

    @Override
    public GameBuilder changeMaximumComputationTimePerMove(final int newMaxComputationTimePerMove) {
        this.maxComputationTimePerMove = newMaxComputationTimePerMove;
        return this;
    }

    @Override
    public KopfundZahlundKanteGameBuilder changeObserverFactoryProvider(
            final ObserverFactoryProvider newObserverFactoryProvider) {
        this.observerFactoryProvider = newObserverFactoryProvider;
        return this;
    }

    @Override
    public KopfundZahlundKanteGame build(final int id) throws GameException, InterruptedException {
        if (!this.firstPlayer.isPresent() || !this.secondPlayer.isPresent()) {
            throw new GameException("A Head and Tail game needs two players.");
        }

        final KopfundZahlundKanteStateImpl initialState = new KopfundZahlundKanteStateImpl(this.firstPlayer.get(),
                this.secondPlayer.get());

        final Map<String, KopfundZahlundKanteStrategy> strategies = new LinkedHashMap<>();
        strategies.put(initialState.getFirstPlayer().getName(), this.firstPlayerStrategy.orElseThrow());
        strategies.put(initialState.getSecondPlayer().getName(), this.secondPlayerStrategy.orElseThrow());
        return new KopfundZahlundKanteGameImpl(
                id,
                initialState,
                strategies,
                this.maxComputationTimePerMove,
                AbstractKopfundZahlundKanteMove.class::isInstance,
                this.observerFactoryProvider);
    }

}
