package de.fhdw.gaming.ipspiel22.kopfundzahlundkante.domain.impl;

import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.PlayerState;
import de.fhdw.gaming.ipspiel22.kopfundzahlundkante.domain.KopfundZahlundKanteAnswerEnum;
import de.fhdw.gaming.ipspiel22.kopfundzahlundkante.domain.KopfundZahlundKantePlayer;
import de.fhdw.gaming.ipspiel22.kopfundzahlundkante.domain.KopfundZahlundKanteState;

/**
 * Implements {@link KopfundZahlundKanteState}.
 */
public final class KopfundZahlundKanteStateImpl implements KopfundZahlundKanteState {

    /**
     * The first player.
     */
    private final KopfundZahlundKantePlayer firstPlayer;

    /**
     * The second player.
     */
    private final KopfundZahlundKantePlayer secondPlayer;

    /**
     * Creates KopfundZahl state.
     *
     * @param firstPlayer  The first player.
     * @param secondPlayer The second player.
     * @throws GameException if the state cannot be created according to the rules of the game.
     */
    public KopfundZahlundKanteStateImpl(final KopfundZahlundKantePlayer firstPlayer,
            final KopfundZahlundKantePlayer secondPlayer) throws GameException {

        this.firstPlayer = Objects.requireNonNull(firstPlayer, "firstPlayer");
        this.secondPlayer = Objects.requireNonNull(secondPlayer, "secondPlayer");

        if (this.firstPlayer.getName().equals(this.secondPlayer.getName())) {
            throw new IllegalArgumentException(
                    String.format("Both players have the same name '%s'.", this.firstPlayer.getName()));
        }
    }

    /**
     * Creates a KopfundZahl state by copying an existing one.
     *
     * @param source The state to copy.
     */
    KopfundZahlundKanteStateImpl(final KopfundZahlundKanteStateImpl source) {
        this.firstPlayer = source.firstPlayer.deepCopy();
        this.secondPlayer = source.secondPlayer.deepCopy();
    }

    /**
     * Returns the first player.
     */
    @Override
    public KopfundZahlundKantePlayer getFirstPlayer() {
        return this.firstPlayer;
    }

    /**
     * Return the second player.
     */
    @Override
    public KopfundZahlundKantePlayer getSecondPlayer() {
        return this.secondPlayer;
    }

    @Override
    public String toString() {
        return String.format(
                "KopfundZahlundKanteState[firstPlayer=%s, secondPlayer=%s]", this.firstPlayer, this.secondPlayer);
    }

    @Override
    public boolean equals(final Object obj) {
        if (obj instanceof KopfundZahlundKanteStateImpl) {
            final KopfundZahlundKanteStateImpl other = (KopfundZahlundKanteStateImpl) obj;
            return this.firstPlayer.equals(other.firstPlayer) && this.secondPlayer.equals(other.secondPlayer);
        }
        return false;
    }

    @Override
    public int hashCode() {
        return Objects.hash(this.firstPlayer, this.secondPlayer);
    }

    @Override
    public KopfundZahlundKanteState deepCopy() {
        return new KopfundZahlundKanteStateImpl(this);
    }

    @Override
    public Map<String, KopfundZahlundKantePlayer> getPlayers() {
        final Map<String, KopfundZahlundKantePlayer> result = new LinkedHashMap<>();
        result.put(this.firstPlayer.getName(), this.firstPlayer);
        result.put(this.secondPlayer.getName(), this.secondPlayer);
        return result;
    }

    @Override
    public Set<KopfundZahlundKantePlayer> computeNextPlayers() {
        final Set<KopfundZahlundKantePlayer> playersWithoutMove = new LinkedHashSet<>();
        if (this.firstPlayer.getAnswer().isEmpty()) {
            playersWithoutMove.add(this.firstPlayer);
        }
        if (this.secondPlayer.getAnswer().isEmpty()) {
            playersWithoutMove.add(this.secondPlayer);
        }
        return playersWithoutMove;
    }

    @Override
    public void nextTurn() {
        final Set<KopfundZahlundKantePlayer> playersWithoutMove = this.computeNextPlayers();
        if (playersWithoutMove.isEmpty()) {
            final KopfundZahlundKanteAnswerEnum answerOfFirstPlayer = this.firstPlayer.getAnswer().orElseThrow();
            final KopfundZahlundKanteAnswerEnum answerOfSecondPlayer = this.secondPlayer.getAnswer().orElseThrow();

            final Double outcomeOfFirstPlayer = this.firstPlayer.getPossibleOutcomes().get(answerOfFirstPlayer)
                    .get(answerOfSecondPlayer);
            this.firstPlayer.setState(outcomeToState(outcomeOfFirstPlayer));
            this.firstPlayer.setOutcome(outcomeOfFirstPlayer);

            final Double outcomeOfSecondPlayer = this.secondPlayer.getPossibleOutcomes().get(answerOfFirstPlayer)
                    .get(answerOfSecondPlayer);
            this.secondPlayer.setState(outcomeToState(outcomeOfSecondPlayer));
            this.secondPlayer.setOutcome(outcomeOfSecondPlayer);
        }
    }

    /**
     * Computes a player state from an outcome.
     *
     * @param outcome The player's outcome.
     */
    private static PlayerState outcomeToState(final Double outcome) {
        return outcome > 0.0 ? PlayerState.WON : PlayerState.LOST;
    }
}
