package de.schereSteinPapier.domain.impl;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import de.fhdw.gaming.core.domain.AbstractPlayer;
import de.schereSteinPapier.domain.SSPPlayer;

/**
 * Implements {@link SSPPlayer}.
 */
public final class SSPPlayerImpl extends AbstractPlayer<SSPPlayer> implements SSPPlayer {

    /**
     * The possible outcomes of this player. The key for the first-level map is the answer of the first player, the key
     * for the second-level map is the answer of the second player.
     */
    private final Map<String, Map<String, Double>> possibleOutcomes;
    /**
     * The answer of the player.
     */
    private Optional<String> answer;

    /**
     * Creates a SSP player.
     *
     * @param name             The name of the player.
     * @param possibleOutcomes The possible outcomes of this player. The key for the first-level map is the answer of
     *                         the first player, the key for the second-level map is the answer of the second player.
     */
    SSPPlayerImpl(final String name, final Map<String, Map<String, Double>> possibleOutcomes) {
        super(name);
        this.possibleOutcomes = Collections
                .unmodifiableMap(new LinkedHashMap<>(Objects.requireNonNull(possibleOutcomes, "possibleOutcomes")));
        this.answer = Optional.empty();
    }

    /**
     * Creates a SSP player.
     *
     * @param source The {@link SSPPlayer} to copy.
     */
    SSPPlayerImpl(final SSPPlayer source) {
        super(source);
        this.possibleOutcomes = source.getPossibleOutcomes();
        this.answer = source.getAnswer();
    }

    @Override
    public String toString() {
        return String
                .format("SSP_Player[name=%s, state=%s, outcome=%s, answer=%s]", this.getName(), this.getState(),
                        this.getOutcome(), this.answer);
    }

    @Override
    public boolean equals(final Object obj) {
        if (obj instanceof SSPPlayerImpl) {
            final SSPPlayerImpl other = (SSPPlayerImpl) obj;
            return super.equals(obj) && this.answer.equals(other.answer);
        }
        return false;
    }

    @Override
    public int hashCode() {
        return super.hashCode() ^ this.answer.hashCode();
    }

    @Override
    public Map<String, Map<String, Double>> getPossibleOutcomes() {
        return this.possibleOutcomes;
    }

    @Override
    public Optional<String> getAnswer() {
        return this.answer;
    }

    @Override
    public void setAnswer(final String newAnswer) {
        if (this.answer.isPresent()) {
            throw new IllegalStateException(String.format("Player %s tried to change her answer.", this.getName()));
        }
        this.answer = Optional.of(newAnswer);
    }

    @Override
    public SSPPlayerImpl deepCopy() {
        return new SSPPlayerImpl(this);
    }
}
