package de.fhdw.gaming.ipspiel22.searchtree.algorithm;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.Move;
import de.fhdw.gaming.core.domain.Player;
import de.fhdw.gaming.core.domain.State;
import de.fhdw.gaming.ipspiel22.searchtree.domain.MinMaxGame;

/**
 * MinMax algorithm implementation.
 *
 * @param <P> The type of player.
 * @param <S> The type of state.
 * @param <M> The type of move.
 * @param <G> The type of object implementing the {@link MinMaxGame} interface.
 */
public final class MinMaxAlgorithm<P extends Player<P>, S extends State<P, S>, M extends Move<P, S>,
        G extends MinMaxGame<P, S, M>> {
    /**
     * Variable for the class with the necessary functions.
     */
    private final G game;

    /**
     * Constructs an object of this class.
     *
     * @param game The interface to the underlying game.
     */
    public MinMaxAlgorithm(final G game) {
        this.game = game;
    }

    /**
     * Returns the best move for the current player.
     *
     * @param depth The search depth.
     */
    public Optional<M> getBestMove(final int depth) throws GameException {
        if (this.game.isGameOver()) {
            return Optional.empty();
        }
        final Map<M, Double> scores = new LinkedHashMap<>();
        for (final M move : this.game.getPossibleMoves()) {
            this.game.commitMove(move);
            scores.put(move, -this.miniMax(depth, Double.NEGATIVE_INFINITY, Double.POSITIVE_INFINITY));
            this.game.rollbackMove(move);
        }
        return scores.entrySet().stream().max((e1, e2) -> e1.getValue().compareTo(e2.getValue()))
                .map(Map.Entry::getKey);
    }

    /**
     * MinMax algorithm which calculates the best next move.
     *
     * @param depth the depth of searching (moves in the future).
     * @param alpha alpha value for alpha-beta pruning (on first call: -inf).
     * @param beta  beta value for alpha-beta pruning (on first call: +inf).
     * @return the value for the next move.
     */
    public double miniMax(final int depth, final double alpha, final double beta) throws GameException {
        if (depth == 0 || this.game.isGameOver()) {
            return this.game.evaluateStateful();
        }
        double maxWert = alpha;
        final List<M> moves = this.game.getPossibleMoves();
        for (final M move : moves) {
            this.game.commitMove(move);
            final double wert = -this.miniMax(depth - 1, -beta, -maxWert);
            this.game.rollbackMove(move);
            if (wert > maxWert) {
                maxWert = wert;
                if (maxWert >= beta) {
                    break;
                }
            }
        }
        return maxWert;
    }
}
