package de.fhdw.gaming.ipspiel22.vierGewinnt.gui.impl;

import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;

import de.fhdw.gaming.ipspiel22.vierGewinnt.domain.VGBoard;
import de.fhdw.gaming.ipspiel22.vierGewinnt.domain.VGField;
import de.fhdw.gaming.ipspiel22.vierGewinnt.domain.VGFieldState;
import de.fhdw.gaming.ipspiel22.vierGewinnt.domain.VGPlayer;
import de.fhdw.gaming.ipspiel22.vierGewinnt.domain.VGPosition;
import de.fhdw.gaming.ipspiel22.vierGewinnt.domain.VGState;
import de.fhdw.gaming.ipspiel22.vierGewinnt.gui.VierGewinntBoardEventProvider;
import de.fhdw.gaming.ipspiel22.vierGewinnt.gui.event.VierGewinntBoardEvent;
import de.fhdw.gaming.ipspiel22.vierGewinnt.gui.event.VierGewinntMakeMoveBoardEvent;
import javafx.application.Platform;
import javafx.scene.Cursor;
import javafx.scene.input.MouseButton;
import javafx.scene.input.MouseEvent;

/**
 * Implements user interaction with a connect-4 board.
 */
public final class VierGewinntBoardEventProviderImpl implements VierGewinntBoardEventProvider {

    /**
     * The associated {@link VierGewinntBoardView}.
     */
    private final VierGewinntBoardView boardView;

    /**
     * Constructor.
     * 
     * @param boardView The associated {@link VierGewinntBoardView}.
     */
    public VierGewinntBoardEventProviderImpl(final VierGewinntBoardView boardView) {
        this.boardView = boardView;
    }
    
    @Override
    public VierGewinntBoardEvent waitForEvent(final VGPlayer player, final VGState state) {
        final Map<VGPosition, VierGewinntFieldView> emptyFieldViews = new LinkedHashMap<>();
        for (final Map.Entry<VGPosition, ? extends VGField> entry : state.getBoard()
                .getFieldsBeing(VGFieldState.EMPTY).entrySet()) {
            emptyFieldViews.put(entry.getKey(), this.boardView.getFieldView(entry.getKey()).orElseThrow());
        }

        final AtomicReference<VierGewinntBoardEvent> event = new AtomicReference<>();
        final Runnable cleanUp;
        if (emptyFieldViews.isEmpty()) {
            final List<VierGewinntFieldView> fieldViews = this.setupInactiveFields(state.getBoard());
            cleanUp = () -> this.cleanUpFields(fieldViews);
        } else {
            this.setupActiveFields(emptyFieldViews, event);
            cleanUp = () -> this.cleanUpFields(emptyFieldViews.values());
        }

        try {
            this.boardView.getUserInputSemaphore().acquire();
            return event.get();
        } catch (final InterruptedException e) {
            return null;
        } finally {
            Platform.runLater(cleanUp);
        }
    }

    /**
     * Sets up the fields when a move is possible (and hence the set of empty fields is not empty).
     *
     * @param emptyFieldViews The non-empty set of views for active fields.
     * @param event           The event to be set when the user selects an active field.
     */
    private void setupActiveFields(final Map<VGPosition, VierGewinntFieldView> emptyFieldViews,
            final AtomicReference<VierGewinntBoardEvent> event) {
        for (final Map.Entry<VGPosition, VierGewinntFieldView> entry : emptyFieldViews.entrySet()) {
            final VGPosition position = entry.getKey();
            final VierGewinntFieldView fieldView = entry.getValue();
            Platform.runLater(() -> {
                fieldView.setCursor(Cursor.CROSSHAIR);
                fieldView.setHighlighted(true);
                fieldView.setOnMouseClicked((final MouseEvent mouseEvent) -> {
                    if (mouseEvent.getButton() == MouseButton.PRIMARY) {
                        event.set(new VierGewinntMakeMoveBoardEvent(position));
                        this.boardView.getUserInputSemaphore().release();
                    }
                });
            });
        }
    }

    /**
     * Sets up the fields when no move is possible (and hence the set of empty fields is empty).
     *
     * @param board The connect-4 board.
     * @return A list of all field views on the board.
     */
    private List<VierGewinntFieldView> setupInactiveFields(final VGBoard board) {
        final List<VGField> fields = new ArrayList<>();
        board.getFields().forEach((final List<? extends VGField> list) -> fields.addAll(list));

        final List<VierGewinntFieldView> fieldViews = new ArrayList<>();
        for (final VGField field : fields) {
            final VierGewinntFieldView fieldView = this.boardView.getFieldView(field.getPosition()).orElseThrow();
            fieldViews.add(fieldView);
            Platform.runLater(() -> {
                fieldView.setCursor(Cursor.CLOSED_HAND);
                fieldView.setOnMouseClicked(null);
            });
        }

        return fieldViews;
    }

    /**
     * Cleans up after user interaction by resetting mouse cursors and removing event handlers.
     *
     * @param fieldViews The modified field views.
     */
    private void cleanUpFields(final Collection<? extends VierGewinntFieldView> fieldViews) {
        for (final VierGewinntFieldView fieldView : fieldViews) {
            fieldView.setCursor(Cursor.DEFAULT);
            fieldView.setHighlighted(false);
            fieldView.setOnMouseClicked(null);
        }
    }

    @Override
    public void cancelWaiting() {
        this.boardView.getUserInputSemaphore().release();
    }

}
