package de.fhdw.gaming.ipspiel22.vierGewinnt.domain.impl;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import de.fhdw.gaming.core.domain.DefaultObserverFactoryProvider;
import de.fhdw.gaming.core.domain.GameBuilder;
import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.ObserverFactoryProvider;
import de.fhdw.gaming.ipspiel22.vierGewinnt.domain.VGGame;
import de.fhdw.gaming.ipspiel22.vierGewinnt.domain.VGGameBuilder;
import de.fhdw.gaming.ipspiel22.vierGewinnt.domain.VGPlayer;
import de.fhdw.gaming.ipspiel22.vierGewinnt.domain.VGPlayerBuilder;
import de.fhdw.gaming.ipspiel22.vierGewinnt.domain.VGStrategy;
import de.fhdw.gaming.ipspiel22.vierGewinnt.moves.impl.AbstractVGMove;

/**
 * Implements {@link VGGameBuilder}.
 */
public class VGGameBuilderImpl implements VGGameBuilder {

    /**
     * The {@link ObserverFactoryProvider}.
     */
    private ObserverFactoryProvider observerFactoryProvider;
    /**
     * The player using red tokens.
     */
    private Optional<VGPlayer> redPlayer;
    /**
     * The strategy of the player using red tokens.
     */
    private Optional<VGStrategy> redPlayerStrategy;
    /**
     * The player using yellow tokens.
     */
    private Optional<VGPlayer> yellowPlayer;

    /**
     * The strategy of the player using yellow tokens.
     */
    private Optional<VGStrategy> yellowPlayerStrategy;
    /**
     * The maximum computation time per move in seconds.
     */
    private int maxComputationTimePerMove;

    /**
     * Creates a Vier gewinnt game builder.
     */
    public VGGameBuilderImpl() {
        this.observerFactoryProvider = new DefaultObserverFactoryProvider();
        this.redPlayer = Optional.empty();
        this.redPlayerStrategy = Optional.empty();
        this.yellowPlayer = Optional.empty();
        this.yellowPlayerStrategy = Optional.empty();
        this.maxComputationTimePerMove = GameBuilder.DEFAULT_MAX_COMPUTATION_TIME_PER_MOVE;
    }

    @Override
    public VGPlayerBuilder createPlayerBuilder() {
        return new VGPlayerBuilderImpl();
    }

    @Override
    public VGGameBuilder addPlayer(final VGPlayer player, final VGStrategy strategy)
            throws GameException {

        if (player.isUsingRedChips() && this.redPlayer.isEmpty()) {
            this.redPlayer = Optional.of(Objects.requireNonNull(player, "player"));
            this.redPlayerStrategy = Optional.of(Objects.requireNonNull(strategy, "redPlayerStrategy"));
        } else if (!player.isUsingRedChips() && this.yellowPlayer.isEmpty()) {
            this.yellowPlayer = Optional.of(Objects.requireNonNull(player, "player"));
            this.yellowPlayerStrategy = Optional.of(Objects.requireNonNull(strategy, "yellowPlayerStrategy"));
        } else {
            throw new GameException(String.format("More than two players are now allowed."));
        }
        return this;
    }

    @Override
    public VGGameBuilder changeMaximumComputationTimePerMove(final int newMaxComputationTimePerMove) {
        this.maxComputationTimePerMove = newMaxComputationTimePerMove;
        return this;
    }

    @Override
    public VGGameBuilder changeObserverFactoryProvider(final ObserverFactoryProvider newObserverFactoryProvider) {
        this.observerFactoryProvider = newObserverFactoryProvider;
        return this;
    }

    @Override
    public VGGame build(final int id) throws GameException, InterruptedException {
        if (!this.redPlayer.isPresent() || !this.yellowPlayer.isPresent()) {
            throw new GameException("A Vier gewinnt game needs two players.");
        }

        final VGBoardImpl board = new VGBoardImpl();
        final VGStateImpl initialState = new VGStateImpl(board, this.redPlayer.get(), this.yellowPlayer.get(), true);

        final Map<String, VGStrategy> strategies = new LinkedHashMap<>();
        strategies.put(initialState.getRedPlayer().getName(), this.redPlayerStrategy.orElseThrow());
        strategies.put(initialState.getYellowPlayer().getName(), this.yellowPlayerStrategy.orElseThrow());
        return new VGGameImpl(
                id,
                initialState,
                strategies,
                this.maxComputationTimePerMove,
                AbstractVGMove.class::isInstance,
                this.observerFactoryProvider);
    }
    
    /**
     * getObserverFactoryProvider.
     * @return
     */
    public ObserverFactoryProvider getObserverFactoryProvider() {
        return observerFactoryProvider;
    }

    /**
     * getMaxComputationTimePerMove.
     * @return
     */
    public int getMaxComputationTimePerMove() {
        return maxComputationTimePerMove;
    }
}
