package de.fhdw.gaming.ipspiel22.kopfundzahl.domain.impl;

import java.util.Collections;
import java.util.Map;
import java.util.Optional;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.ipspiel22.kopfundzahl.domain.KopfundZahlPlayer;
import de.fhdw.gaming.ipspiel22.kopfundzahl.domain.KopfundZahlPlayerBuilder;

/**
 * Implements {@link KopfundZahlPlayerBuilder}.
 */
public final class KopfundZahlPlayerBuilderImpl implements KopfundZahlPlayerBuilder {

    /**
     * The name of the player.
     */
    private Optional<String> name;

    /**
     * The possible outcomes of this player. The key for the first-level map is the answer of the first player, the key
     * for the second-level map is the answer of the second player.
     */
    private Optional<Map<Boolean, Map<Boolean, Double>>> possibleOutcomes;

    /**
     * Creates an {@link KopfundZahlPlayerBuilderImpl}.
     */
    public KopfundZahlPlayerBuilderImpl() {
        this.name = Optional.empty();
        this.possibleOutcomes = Optional.empty();
    }

    @Override
    public KopfundZahlPlayerBuilderImpl changeName(final String newName) {
        this.name = Optional.of(newName);
        return this;
    }

    @Override
    public KopfundZahlPlayerBuilder changePossibleOutcomes(final Map<Boolean, Map<Boolean,
            Double>> newpossibleOutcomes) {
        this.possibleOutcomes = Optional.of(newpossibleOutcomes);
        return this;
    }

    @Override
    public KopfundZahlPlayer build() throws GameException {
        return new KopfundZahlPlayerImpl(
                this.name.orElseThrow(),
                this.checkPossibleOutcomes(this.possibleOutcomes.orElseThrow()));
    }

    /**
     * Checks if all possible outcomes are defined for a player.
     *
     * @param outcomes The possible outcomes for the player.
     */
    private Map<Boolean, Map<Boolean, Double>> checkPossibleOutcomes(
            final Map<Boolean, Map<Boolean, Double>> outcomes) {
        this.checkPossibleOutcome(outcomes, false, false);
        this.checkPossibleOutcome(outcomes, false, true);
        this.checkPossibleOutcome(outcomes, true, false);
        this.checkPossibleOutcome(outcomes, true, true);
        return outcomes;
    }

    /**
     * Checks if a given outcome is defined for a player.
     *
     * @param outcomes     The possible outcomes for the player.
     * @param firstChoice  The choice of the first player.
     * @param secondChoice The choice of the second player.
     */
    private void checkPossibleOutcome(final Map<Boolean, Map<Boolean, Double>> outcomes, final boolean firstChoice,
            final boolean secondChoice) {
        if (outcomes.getOrDefault(firstChoice, Collections.emptyMap()).get(secondChoice) == null) {
            throw new IllegalArgumentException(
                    String.format(
                            "No outcome defined for player '%s' and combination %s/%s.",
                            this.name,
                            toAnswer(firstChoice),
                            toAnswer(secondChoice)));
        }
    }

    /**
     * Maps a boolean value to a "head" or "tail" answer.
     *
     * @param value The value to be mapped.
     */
    private static String toAnswer(final boolean value) {
        return value ? "head" : "tail";
    }

}
