/*
 * Copyright © 2021-2023 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel23-Ssp.
 *
 * Ipspiel23-Ssp is free software: you can redistribute it and/or modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * Ipspiel23-Ssp is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with ipspiel23-Ssp. If not, see
 * <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel23.ssp.domain.impl;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.ipspiel23.ssp.domain.SspPlayer;
import de.fhdw.gaming.ipspiel23.ssp.domain.SspPlayerBuilder;
import de.fhdw.gaming.ipspiel23.ssp.domain.impl.outcomes.SspAnswer;

import java.util.Collections;
import java.util.Map;
import java.util.Optional;

/**
 * Implements {@link SspPlayerBuilder}.
 */
final class SspPlayerBuilderImpl implements SspPlayerBuilder {

    /**
     * The name of the player.
     */
    private Optional<String> name;
    /**
     * The possible outcomes of this player. The key for the first-level map is the answer of the first player, the key
     * for the second-level map is the answer of the second player.
     */
    private Optional<Map<SspAnswer, Map<SspAnswer, Double>>> possibleOutcomes;

    /**
     * Creates an {@link SspPlayerBuilderImpl}.
     */
    SspPlayerBuilderImpl() {
        this.name = Optional.empty();
        this.possibleOutcomes = Optional.empty();
    }

    @Override
    public SspPlayerBuilderImpl changeName(final String newName) {
        this.name = Optional.of(newName);
        return this;
    }

    @Override
    public SspPlayerBuilder changePossibleOutcomes(
            final Map<SspAnswer, Map<SspAnswer, Double>> newPossibleOutcomes) {
        this.possibleOutcomes = Optional.of(newPossibleOutcomes);
        return this;
    }

    @Override
    public SspPlayer build() throws GameException {
        return new SspPlayerImpl(
                this.name.orElseThrow(),
                this.checkPossibleOutcomes(this.possibleOutcomes.orElseThrow()));
    }

    /**
     * Checks if all possible outcomes are defined for a player.
     *
     * @param outcomes The possible outcomes for the player.
     */
    private Map<SspAnswer, Map<SspAnswer, Double>> checkPossibleOutcomes(
            final Map<SspAnswer, Map<SspAnswer, Double>> outcomes) {

        this.checkPossibleOutcome(outcomes, SspAnswer.PAPER, SspAnswer.PAPER);
        this.checkPossibleOutcome(outcomes, SspAnswer.PAPER, SspAnswer.STONE);
        this.checkPossibleOutcome(outcomes, SspAnswer.PAPER, SspAnswer.SCISSORS);

        this.checkPossibleOutcome(outcomes, SspAnswer.STONE, SspAnswer.STONE);
        this.checkPossibleOutcome(outcomes, SspAnswer.STONE, SspAnswer.PAPER);
        this.checkPossibleOutcome(outcomes, SspAnswer.STONE, SspAnswer.SCISSORS);

        this.checkPossibleOutcome(outcomes, SspAnswer.SCISSORS, SspAnswer.STONE);
        this.checkPossibleOutcome(outcomes, SspAnswer.SCISSORS, SspAnswer.PAPER);
        this.checkPossibleOutcome(outcomes, SspAnswer.SCISSORS, SspAnswer.SCISSORS);

        return outcomes;
    }

    /**
     * Checks if a given outcome is defined for a player.
     *
     * @param outcomes     The possible outcomes for the player.
     * @param firstChoice  The choice of the first player.
     * @param secondChoice The choice of the second player.
     */
    private void checkPossibleOutcome(final Map<SspAnswer, Map<SspAnswer, Double>> outcomes,
            final SspAnswer firstChoice,
            final SspAnswer secondChoice) {
        if (outcomes.getOrDefault(firstChoice, Collections.emptyMap()).get(secondChoice) == null) {
            throw new IllegalArgumentException(
                    String.format(
                            "No outcome defined for player '%s' and combination %s/%s.",
                            this.name,
                            firstChoice.toString(),
                            secondChoice.toString()));
        }
    }
}
