package de.fhdw.gaming.ipspiel23.c4.gui.impl;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Player;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4State;
import de.fhdw.gaming.ipspiel23.c4.gui.C4BoardEventProvider;
import de.fhdw.gaming.ipspiel23.c4.gui.event.C4BoardEventVisitor;
import de.fhdw.gaming.ipspiel23.c4.gui.event.C4MakeMoveBoardEvent;
import de.fhdw.gaming.ipspiel23.c4.moves.IC4Move;
import de.fhdw.gaming.ipspiel23.c4.moves.factory.IC4MoveFactory;
import de.fhdw.gaming.ipspiel23.c4.strategies.IC4Strategy;

import java.util.Optional;
import java.util.concurrent.atomic.AtomicReference;

/**
 * Implements {@link IC4Strategy} by asking the user for a move.
 */
public class C4InteractiveStrategy implements IC4Strategy {
    
    /**
     * The factory for creating Connect Four moves.
     */
    private final IC4MoveFactory moveFactory;

    /**
     * Creates an {@link C4InteractiveStrategy}.
     *
     * @param moveFactory The factory for creating Connect Four moves.
     */
    C4InteractiveStrategy(final IC4MoveFactory moveFactory) {
        this.moveFactory = moveFactory;
    }
    /**
     * Asks the user to choose a move.
     */
    @Override
    public Optional<IC4Move> computeNextMove(final int gameId, final IC4Player player,
            final IC4State state) throws GameException {

        final Optional<C4BoardEventProvider> provider = C4GuiObserverImpl.getEventProvider(gameId);
        if (provider.isEmpty()) {
            return Optional.empty();
        }

        final AtomicReference<IC4Move> move = new AtomicReference<>();
        provider.get().waitForEvent(player, state).accept(new C4BoardEventVisitor() {

            @Override
            public void handleMakeMove(final C4MakeMoveBoardEvent event) {
                move.setPlain(
                        C4InteractiveStrategy.this.moveFactory
                                .createMove(player, event.getFieldPosition()));
            }
        });

        return Optional.ofNullable(move.getPlain());
    }

    @Override
    public String toString() {
        return C4InteractiveStrategy.class.getSimpleName();
    }

    @Override
    public boolean isInteractive() {
        return true;
    }

    @Override
    public void abortRequested(final int gameId) {
        final Optional<C4BoardEventProvider> provider = C4GuiObserverImpl.getEventProvider(gameId);
        provider.ifPresent(C4BoardEventProvider::cancelWaiting);
    }
}
