package de.fhdw.gaming.ipspiel23.c4.domain.impl;

import java.util.Arrays;

import de.fhdw.gaming.ipspiel23.c4.domain.C4Direction;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Field;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Position;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Solution;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Player;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4SolutionSlim;

/**
 * A naive implementation of {@link IC4Solution}.
 */
public class C4SolutionHeavy implements IC4Solution {

    /**
     * The player who placed the solution.
     */
    private final IC4Player owner;

    /**
     * The start position of the solution.
     */
    private final IC4Position startPosition;

    /**
     * The end position of the solution.
     */
    private final IC4Position endPosition;

    /**
     * The direction of the solution.
     */
    private final C4Direction direction;

    /**
     * The fields in which the solution is contained.
     */
    private final IC4Field[] containingFields;
    
    /**
     * Creates a new solution.
     * 
     * @param owner The player who placed the solution.
     * @param startPosition The start position of the solution.
     * @param endPosition The end position of the solution.
     * @param direction The direction of the solution.
     * @param containingFields The fields in which the solution is contained.
     */
    public C4SolutionHeavy(final IC4Player owner, final IC4Position startPosition, 
            final IC4Position endPosition, final C4Direction direction,
            final IC4Field... containingFields) {
        this.owner = owner;
        this.startPosition = startPosition;
        this.endPosition = endPosition;
        this.direction = direction;
        this.containingFields = Arrays.copyOf(containingFields, containingFields.length);
    }

    @Override
    public IC4Player getOwner() {
        return this.owner;
    }

    @Override
    public IC4Position getStartPosition() {
        return this.startPosition;
    }

    @Override
    public IC4Position getEndPosition() {
        return this.endPosition;
    }

    @Override
    public C4Direction getDirection() {
        return this.direction;
    }

    @Override
    public int size() {
        return this.containingFields.length;
    }

    @Override
    public IC4Field[] getContainingFields() {
        return Arrays.copyOf(this.containingFields, this.containingFields.length);
    }

    @Override
    public boolean equals(final Object other) {
        if (other == null) {
            return false;
        }
        if (!(other instanceof IC4SolutionSlim)) {
            return false;
        }
        final IC4SolutionSlim otherSolution = (IC4SolutionSlim) other;
        return this.getOwner().equals(otherSolution.getOwner()) 
            // we don't care what direction a solution is facing.
            // the direction itself should probably be internal, but I guess
            // it may have valid use cases for consumers of our library :)
            && (this.getStartPosition().equals(otherSolution.getStartPosition()) 
                    && this.getEndPosition().equals(otherSolution.getEndPosition())
                || this.getStartPosition().equals(otherSolution.getEndPosition()) 
                    && this.getEndPosition().equals(otherSolution.getStartPosition()))
            && this.size() == otherSolution.size();
    }

    @Override
    public int hashCode() {
        int hash = 7 * 31;
        hash += this.getOwner().hashCode();
        hash = hash * 31 + (this.getStartPosition().hashCode() ^ this.getEndPosition().hashCode());
        hash = hash * 31 + this.size();
        return hash;
    }

    @Override
    public String toString() {
        return String.format("C4SolutionSlim: (%d, %d) -- %d -- (%d, %d)",
            this.getStartPosition().getRow(), this.getStartPosition().getColumn(), 
                this.getOwner().getToken(), this.getEndPosition().getRow(), this.getEndPosition().getColumn());
    }
}
