package de.fhdw.gaming.ipspiel23.c4.domain.impl.validation;

import de.fhdw.gaming.core.ui.InputProviderException;

/**
 * Represents the absolute limits, and currently set dimensions of a Connect-N board.
 * User-provided values are stored in this class, and validated against the limits.
 */
public class C4BoardLimits {

    /**
     * The minimum number of rows a board must have.
     */
    // for the incredible "Connect Two" experience where the first player wins by default after the second move :)
    public static final int MIN_ROW_COUNT = 2;

    /**
     * The minimum number of columns a board must have.
     */
    // for the incredible "Connect Two" experience where the first player wins by default after the second move :)
    public static final int MIN_COLUMN_COUNT = 2;

    /**
     * The maximum number of rows a board can have.
     */
    // these are theoretical maximums. In the end, actualColumnCount * actualRowCount must be <= MAX_FIELD_COUNT
    public static final int MAX_ROW_COUNT = Integer.MAX_VALUE / MIN_COLUMN_COUNT;

    /**
     * The maximum number of columns a board can have.
     */
    // these are theoretical maximums. In the end, actualColumnCount * actualRowCount must be <= MAX_FIELD_COUNT
    public static final int MAX_COLUMN_COUNT = Integer.MAX_VALUE / MIN_ROW_COUNT;

    /**
     * The maximum number of fields a board can have.
     */
    // we use 32 bit indices for our internal token-state array, so if more than 4 * 2^31 - 1 = 8 GiB of RAM are 
    // available we can use the whole 31 bit range for indexing (also: why doesn't java have unsigned integers???).
    // fun fact:
    // at 1 second per move a draw could actually happen within our life time at
    // just over 68 Years of 24/7 exciting Connect-N gameplay :)
    public static final int MAX_FIELD_COUNT = Integer.MAX_VALUE;

    /**
     * The minimum allowed number that can be set as the number of tokens that need to be matched to form a solution.
     */
    // "Connect One" wouldn't be too exciting TBH. Also I don't think our board-state-validation algorithm would 
    // support it.
    public static final int MIN_SOLUTION_SIZE = 2;

    /**
     * The maximum allowed number that can be set as the number of tokens that need to be matched to form a solution.
     */
    // Its probably a good idea to keep the solution size below the maximum theoretical board boundaries
    public static final int MAX_SOLUTION_SIZE = Math.max(MAX_ROW_COUNT, MAX_COLUMN_COUNT);

    /**
     * The minimum number of players that can play on a board.
     */
    // to me it's questionable how exciting a Connect Four game with a single player would be,
    // but who am I to judge what gets other people thrilled ¯\_(ツ)_/¯
    public static final int MIN_NUMBER_OF_PLAYERS = 1;

    /**
     * The maximum number of players that can play on a board.
     */
    // let's assume we actually want each player to have at least the theoretical possibility to win
    // fun fact:
    // at the theoretical maximum we would still would support 33.5 times more players than RGB24 has colors available
    // to differentiate between the pieces :P
    // so the GUI team would have to start playing around with the alpha channel in RGB32 to give every player a unique
    // color.
    // also: assuming 16:9 aspect ratio you'd have to own at least a 30897x17379 px monitor to display the whole board
    // at once, so a 32k display would *just* not suffice at "only" 30720x17280 px :)
    public static final int MAX_NUMBER_OF_PLAYERS = C4BoardLimits.MAX_FIELD_COUNT / C4BoardLimits.MIN_SOLUTION_SIZE;

    /**
     * The currently specified board dimensions.
     */
    private final C4BoardDimensions boardDimensions;

    /**
     * The currently specified number of tokens that need to be matched to form a solution.
     */
    private int solutionSize;

    /**
     * The total number of players that will play on the board.
     */
    private final int playerCount;

    /**
     * Creates a new instance of {@link C4BoardLimits}.
     * @param playerCount the actual number of players that will play on the board
     * @param initialRowCount the initial number of rows the board will have
     * @param initialColumnCount the initial number of columns the board will have
     * @param initialSolutionSize the initial number of tokens that need to be matched to form a solution
     */
    public C4BoardLimits(final int playerCount, final int initialRowCount,
            final int initialColumnCount, final int initialSolutionSize) {
        this.playerCount = playerCount;
        this.boardDimensions = new C4BoardDimensions(initialRowCount, initialColumnCount);
        this.solutionSize = initialSolutionSize;
    }

    /**
     * Returns the number of players that will play on the board.
     */
    public int getPlayerCount() {
        return playerCount;
    }

    /**
     * Returns the number of rows the board will have.
     */
    public int getRowCount() {
        return this.boardDimensions.getRowCount();
    }

    /**
     * Updates the number of rows the board will have to the specified value.
     * @param rowCount the new number of rows the board will have
     */
    public void setRowCount(final int rowCount) {
        this.boardDimensions.setRowCount(rowCount);
    }

    /**
     * Returns the number of columns the board will have.
     */
    public int getColumnCount() {
        return this.boardDimensions.getColumnCount();
    }

    /**
     * Updates the number of columns the board will have to the specified value.
     * @param columnCount the new number of columns the board will have
     */
    public void setColumnCount(final int columnCount) {
        this.boardDimensions.setColumnCount(columnCount);
    }

    /**
     * Returns the number of tokens that need to be matched to form a solution.
     */
    public int getSolutionSize() {
        return solutionSize;
    }

    /**
     * Updates the number of tokens that need to be matched to form a solution to the specified value.
     * @param solutionSize the new number of tokens that need to be matched to form a solution
     */
    public void setSolutionSize(final int solutionSize) {
        this.solutionSize = solutionSize;
    }

    /**
     * Checks if the current board settings comply with the board limits.
     * @throws InputProviderException if the current board limits are invalid
     */
    public void assertIsValid() throws InputProviderException {
        // validate board size
        this.assertStaticLimits();
        this.boardDimensions.assertStaticLimits();

        final int fieldCount = this.boardDimensions.getRowCount() * this.boardDimensions.getColumnCount();
        // check for fieldCount > MAX_FIELD_COUNT
        if (fieldCount < 0) { // 32 bit signed integer overflow
            throw new InputProviderException("The board is too large: the total number of fields exceeds "
                + "the maximum allowed value (" + MAX_FIELD_COUNT + ").");
        }

        // validate solution size
        // does the solution fit on the board?
        if (this.solutionSize > Math.max(this.boardDimensions.getRowCount(), this.boardDimensions.getColumnCount())) {
            throw new InputProviderException("The required solution size is too large: the solution size exceeds "
                + "the number of rows and columns on the board. This is dumb.");
        }
        // has every player the possibility to win?
        if (fieldCount < this.playerCount * this.solutionSize) {
            throw new InputProviderException("The current board settings do not allow every player to win: "
                + "the number of fields is too small to allow every player to form a solution, if players "
                + "take turns placing tokens.");
        }
    }
    
    /**
     * Checks if the current board settings comply with the static board limits.
     * @throws InputProviderException if they don't comply...
     */
    private void assertStaticLimits() throws InputProviderException {
        if (this.solutionSize < MIN_SOLUTION_SIZE) {
            throw new InputProviderException("The specified solution size is too small: the solution size "
                + "must be at least " + MIN_SOLUTION_SIZE + ".");
        }
        if (this.solutionSize > MAX_SOLUTION_SIZE) {
            throw new InputProviderException("The specified solution size is too large: the solution size "
                + "must not exceed " + MAX_SOLUTION_SIZE + ".");
        }
    }

    @Override
    public boolean equals(final Object object) {
        if (object == null) {
            return false;
        }
        if (object == this) {
            return true;
        }
        if (!(object instanceof C4BoardLimits)) {
            return false;
        }
        final C4BoardLimits other = (C4BoardLimits) object;
        return this.boardDimensions.equals(other.boardDimensions)
            && this.solutionSize == other.solutionSize
            && this.playerCount == other.playerCount;
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = hash * 31 + this.boardDimensions.hashCode();
        hash = hash * 31 + this.solutionSize;
        hash = hash * 31 + this.playerCount;
        return hash;
    }
}
