package de.fhdw.gaming.ipspiel23.c4.domain;

import de.fhdw.gaming.ipspiel23.c4.domain.impl.C4Position;

/**
 * Represents a direction on the connect four board relative to a {@link C4Position}.
 */
public enum C4Direction {
    /**
     * The direction to the top.
     */
    NORTH {
        @Override
        public int stepFromColumn(final int column, final int stepCount) {
            return column;
        }

        @Override
        public int stepFromRow(final int row, final int stepCount) {
            return row - stepCount;
        }

        @Override
        public C4Direction getInverse() {
            return SOUTH;
        }
    },
    /**
     * The direction to the top right.
     */
    NORTH_EAST {
        @Override
        public int stepFromColumn(final int column, final int stepCount) {
            return column + stepCount;
        }

        @Override
        public int stepFromRow(final int row, final int stepCount) {
            return row - stepCount;
        }

        @Override
        public C4Direction getInverse() {
            return SOUTH_WEST;
        }
    },
    /**
     * The direction to the right.
     */
    EAST {
        @Override
        public int stepFromColumn(final int column, final int stepCount) {
            return column + stepCount;
        }

        @Override
        public int stepFromRow(final int row, final int stepCount) {
            return row;
        }

        @Override
        public C4Direction getInverse() {
            return WEST;
        }
    },
    /**
     * The direction to the bottom right.
     */
    SOUTH_EAST {
        @Override
        public int stepFromColumn(final int column, final int stepCount) {
            return column + stepCount;
        }

        @Override
        public int stepFromRow(final int row, final int stepCount) {
            return row + stepCount;
        }

        @Override
        public C4Direction getInverse() {
            return NORTH_WEST;
        }
    },
    /**
     * The direction to the bottom.
     */
    SOUTH {
        @Override
        public int stepFromColumn(final int column, final int stepCount) {
            return column;
        }

        @Override
        public int stepFromRow(final int row, final int stepCount) {
            return this.getInverse().stepFromRow(row, -stepCount);
        }

        @Override
        public C4Direction getInverse() {
            return NORTH;
        }
    },
    /**
     * The direction to the bottom left.
     */
    SOUTH_WEST {
        @Override
        public int stepFromColumn(final int column, final int stepCount) {
            return this.getInverse().stepFromColumn(column, -stepCount);
        }

        @Override
        public int stepFromRow(final int row, final int stepCount) {
            return this.getInverse().stepFromRow(row, -stepCount);
        }

        @Override
        public C4Direction getInverse() {
            return NORTH_EAST;
        }
    },
    /**
     * The direction to the left.
     */
    WEST {
        @Override
        public int stepFromColumn(final int column, final int stepCount) {
            return this.getInverse().stepFromColumn(column, -stepCount);
        }

        @Override
        public int stepFromRow(final int row, final int stepCount) {
            return row;
        }

        @Override
        public C4Direction getInverse() {
            return EAST;
        }
    },
    /**
     * The direction to the top left.
     */
    NORTH_WEST {
        @Override
        public int stepFromColumn(final int column, final int stepCount) {
            return this.getInverse().stepFromColumn(column, -stepCount);
        }

        @Override
        public int stepFromRow(final int row, final int stepCount) {
            return this.getInverse().stepFromRow(row, -stepCount);
        }

        @Override
        public C4Direction getInverse() {
            return SOUTH_EAST;
        }
    };
    
    /**
     * Steps the specified number of fields from the specified column in this direction.
     * @param column The column index to step from.
     * @param stepCount The number of fields to step.
     * @return The resulting column index.
     */
    public abstract int stepFromColumn(int column, int stepCount);
    
    /**
     * Steps the specified number of fields from the specified row in this direction.
     * @param row The row index to step from.
     * @param stepCount The number of fields to step.
     * @return The resulting row index.
     */
    public abstract int stepFromRow(int row, int stepCount);

    /**
     * Steps the specified number of fields from the specified position in this direction.
     * @param position The position to step from.
     * @param stepCount The number of fields to step.
     * @return The resulting position.
     */
    public IC4Position stepFrom(final IC4Position position, final int stepCount) {
        final int resultRow = stepFromRow(position.getRow(), stepCount);
        final int resultColumn = stepFromColumn(position.getColumn(), stepCount);
        return new C4Position(resultRow, resultColumn);
    }
    
    /**
     * Gets the inverse direction of this direction.
     * @return The inverse direction.
     */
    public abstract C4Direction getInverse();
}
