package de.fhdw.gaming.ipspiel23.c4.domain.impl;

import java.util.Optional;

import de.fhdw.gaming.ipspiel23.c4.domain.C4Direction;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Board;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Position;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Field;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Player;

/**
 * The default implementation of {@link IC4Field}.
 */
public class C4FieldHeavy implements IC4Field {

    /**
     * The parent board that contains this field.
     */
    private final IC4Board board;

    /**
     * The lazily initialized {@link IC4Position} instance that represents the position of this field.
     */
    private final IC4Position position;

    /**
     * The player who occupies the field.
     */
    private Optional<IC4Player> occupyingPlayer;

    /**
     * Creates a new instance of {@link C4Field}.
     * 
     * @param board The parent board that contains this field.
     * @param position The position of this field.
     */
    public C4FieldHeavy(final IC4Board board, final IC4Position position) {
        this.board = board;
        this.position = position;
        this.occupyingPlayer = Optional.empty();
    }

    @Override
    public IC4Board getBoard() {
        return this.board;
    }

    @Override
    public IC4Position getBoardPosition() {
        return this.position;
    }

    @Override
    public Optional<IC4Player> getOccupyingPlayer() {
        return this.occupyingPlayer;
    }

    @Override
    public boolean trySetOccupyingPlayer(final IC4Player player, final boolean allowOverride) {
        if (this.occupyingPlayer.isPresent() && !allowOverride) {
            return false;
        }
        this.occupyingPlayer = Optional.ofNullable(player);
        return true;
    }

    @Override
    public boolean hasNeighbor(final C4Direction direction) {
        final IC4Position neighborPosition = direction.stepFrom(this.getBoardPosition(), 1);
        return this.getBoard().checkBounds(neighborPosition);
    }

    @Override
    public IC4Field getNeighbor(final C4Direction direction) {
        return getNeighborInternal(direction, true);
    }

    @Override
    public Optional<IC4Field> tryGetNeighbor(final C4Direction direction) {
        final IC4Field neighbor = getNeighborInternal(direction, false);
        return Optional.ofNullable(neighbor);
    }

    /**
     * Gets the neighbor of this field in the provided direction.
     * 
     * @param direction The direction to get the neighbor in.
     * @param throwOob Whether to throw an {@link IndexOutOfBoundsException} if the neighbor is 
     * out of bounds, or to return null.
     * @return The neighbor of this field in the provided direction, or null if the neighbor is 
     * out of bounds and throwOob is false.
     */
    private IC4Field getNeighborInternal(final C4Direction direction, final boolean throwOob) {
        final IC4Position neighborPosition = direction.stepFrom(this.getBoardPosition(), 1);
        
        if (!this.getBoard().checkBounds(neighborPosition)) {
            if (throwOob) {
                throw new IndexOutOfBoundsException("The provided direction violates the bounds of the game board.");
            }
            return null;
        }
        return this.getBoard().tryGetField(neighborPosition).orElseThrow();
    }

    @Override
    public boolean equals(final Object obj) {
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof IC4Field)) {
            return false;
        }
        final IC4Field other = (IC4Field) obj;
        return this.getBoardPosition().equals(other.getBoardPosition()) 
            && this.getBoard().equals(other.getBoard());
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = hash * 31 + this.getBoardPosition().hashCode();
        hash = hash * 31 + this.getBoard().hashCode();
        return hash;
    }

    @Override
    public String toString() {
        final StringBuilder sb = new StringBuilder(32);
        sb.append("C4FieldHeavy[player=").append(this.getOccupyingPlayer().map(IC4Player::getName).orElse("empty"))
            .append(", position=").append(this.getBoardPosition())
            .append(']');
        return sb.toString();
    }

    @Override
    public IC4Field deepCopy() {
        return new C4FieldHeavy(this.board, this.position);
    }
}
