package de.fhdw.gaming.ipspiel23.c4.domain.impl.evaluation;

import java.util.Set;

import de.fhdw.gaming.ipspiel23.c4.domain.C4Direction;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4SolutionSlim;
import de.fhdw.gaming.ipspiel23.c4.domain.impl.C4BoardSlim;
import de.fhdw.gaming.ipspiel23.c4.domain.impl.C4SolutionSlim;

import static de.fhdw.gaming.ipspiel23.c4.domain.impl.C4BoardSlim.EMPTY_TOKEN;

/**
 * A {@link C4SolutionAnalyzer} that analyzes the board on the diagonal from the top left to the bottom right.
 */
public class C4SolutionAnalyzerDiagonalRight extends C4SolutionAnalyzerDiagonal {

    /**
     * Creates a new {@link C4SolutionAnalyzerDiagonalRight}.
     * @param board The board to analyze.
     */
    public C4SolutionAnalyzerDiagonalRight(final C4BoardSlim board) {
        super(board, C4Direction.NORTH_WEST);
    }

    @Override
    public IC4SolutionSlim tryFindFirstSolution(final IC4SolutionSlim currentSolution, final boolean updateCache) {
        // "Assignment of parameter 'currentSolution' is not allowed." :P
        IC4SolutionSlim solution = currentSolution;
        
        int iteration = 0;
        // determine start row of the last column (skip first targetCount rows as the will never contain a solution)
        for (int startRow = targetCount(); solution == null && startRow < rowMax(); startRow++, iteration++) {
            if (mayContainSolution(iteration)) {
                solution = tryFindFirstSolution(startRow, colMax() - 1, iteration, updateCache);
            }
        }
        // determine start column of last row (ignore first targetCount columns as the will never contain a solution)
        // don't evaluate diagonal starting in bottom right twice (startCol = colMax - 2)
        for (int startCol = colMax() - 2; solution == null && startCol >= targetCount(); startCol--, iteration++) {
            if (mayContainSolution(iteration)) {
                solution = tryFindFirstSolution(rowMax() - 1, startCol, iteration, updateCache);
            }
        }
        return solution;
    }
    
    /**
     * Lazily searches for the first solution on the board starting at the specified position.
     * @param startRow the row to start at
     * @param startCol the column to start at
     * @param iteration the current iteration of the search
     * @param updateCache whether to update the solution cache, preventing the same line to be checked again
     * @return The first solution on the board or null if no solution was found.
     */
    private IC4SolutionSlim tryFindFirstSolution(final int startRow, final int startCol, 
            final int iteration, final boolean updateCache) {
        int count = 0;
        int lastToken = 0;
        boolean isDiagonalFull = true;
        // "ForLoopVariableCount: Too many control variables in the for statement" :P
        int row = startRow;
        int col = startCol;
        for (; row >= 0 && col >= 0; row--, col--) {
            final int token = board().getTokenUnsafe(row, col);
            count = countConsecutivesBranchless(count, token, lastToken);
            if (count >= targetCount()) {
                return scanRemaining(token, row, col);
            }
            isDiagonalFull &= token != EMPTY_TOKEN;
            lastToken = token;
        }
        if (updateCache && isDiagonalFull) {
            noSolutionIn(iteration);
        }
        return null;
    }

    @Override
    public void findAllSolutions(final Set<IC4SolutionSlim> resultSet, final boolean updateCache) {
        int iteration = 0;
        // determine start row of the last column (skip first targetCount rows)
        for (int startRow = targetCount(); startRow < rowMax(); startRow++, iteration++) {
            if (mayContainSolution(iteration)) {
                findAllSolutions(resultSet, startRow, colMax() - 1, iteration, updateCache);
            }
        }
        // determine start column of last row (ignore first targetCount columns)
        // don't evaluate diagonal starting in bottom right twice (startCol = colMax - 2)
        for (int startCol = colMax() - 2; startCol >= targetCount(); startCol--, iteration++) {
            if (mayContainSolution(iteration)) {
                findAllSolutions(resultSet, rowMax() - 1, startCol, iteration, updateCache);
            }
        }
    }
    
    /**
     * Eagerly searches for all solutions on the board starting at the specified position.
     * @param resultSet the set to which any solutions are added
     * @param startRow the row to start at
     * @param startCol the column to start at
     * @param iteration the current iteration of the search
     * @param updateCache whether to update the solution cache, preventing the same line to be checked again
     */
    private void findAllSolutions(final Set<IC4SolutionSlim> resultSet, final int startRow,
            final int startCol, final int iteration, final boolean updateCache) {
        int count = 0;
        int lastToken = 0;
        boolean isDiagonalFull = true;
        boolean diagonalContainsSolution = false;
        // "ForLoopVariableCount: Too many control variables in the for statement" :P
        int row = startRow;
        int col = startCol;
        for (; row >= 0 && col >= 0; row--, col--) {
            final int token = board().getTokenUnsafe(row, col);
            count = countConsecutivesBranchless(count, token, lastToken);
            if (count >= targetCount()) {
                count = 0;
                final C4SolutionSlim solution = scanRemaining(token, row, col);
                resultSet.add(solution);
                diagonalContainsSolution = true;
                // skip to end of solution
                row = solution.getRowIndexEnd();
                col = solution.getColumnIndexEnd();
            }
            isDiagonalFull &= token != EMPTY_TOKEN;
            lastToken = token;
        }
        if (updateCache && isDiagonalFull && !diagonalContainsSolution) {
            noSolutionIn(iteration);
        }
    }

    @Override
    protected C4SolutionSlim scanRemaining(final int token, final int startRow, final int startCol) {
        int row = startRow - 1;
        int col = startCol - 1;
        while (row >= 0 && col >= 0 && board().getTokenUnsafe(row, col) == token) {
            row--; 
            col--;
        }
        // revert last change to get inclusive upper bound
        row++;
        col++;
        return solutionOf(token, row, col, board().getMinimumSolutionSize() + startRow - row);
    }
}
