package de.fhdw.gaming.ipspiel23.dilemma.strategy.internals.generous_tit_for_tat;


import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.ipspiel23.dilemma.domain.DilemmaAnswerType;
import de.fhdw.gaming.ipspiel23.dilemma.domain.IDilemmaPlayer;
import de.fhdw.gaming.ipspiel23.dilemma.domain.IDilemmaState;
import de.fhdw.gaming.ipspiel23.dilemma.moves.IDilemmaMove;
import de.fhdw.gaming.ipspiel23.dilemma.moves.IDilemmaMoveFactory;
import de.fhdw.gaming.ipspiel23.dilemma.strategy.internals.DilemmaMemoryStrategy;
import de.fhdw.gaming.ipspiel23.dilemma.strategy.internals.DilemmaRoundData;
import de.fhdw.gaming.ipspiel23.memory.GameMemoryCapacity;
import de.fhdw.gaming.ipspiel23.memory.IGameMemory;
import de.fhdw.gaming.ipspiel23.memory.IGameMemoryCapacity;

import java.util.Map;
import java.util.Optional;
import java.util.Random;

/**
 * Cooperates on the first round and after the opponent cooperated.
 * After a defection, it cooperates with a probability p = min(1 - (T-R/R-S), (R-P/T-P))
 */
public class DilemmaGenerousTitForTatStrategy extends DilemmaMemoryStrategy {
    
    /**
     * Used to generate stream of pseudorandom numbers: number generator.
     */
    private static final Random RANDOM = new Random();

    /**
     * Creates a new instance of {@link DilemmaGenerousTitForTatStrategy}.
     *
     * @param moveFactory the move factory to use
     */
    protected DilemmaGenerousTitForTatStrategy(final IDilemmaMoveFactory moveFactory) {
        super(moveFactory);
    }

    @Override
    protected IGameMemoryCapacity requestedMemoryCapacity() {
        return GameMemoryCapacity.of(1);
    }


    @Override
    public Optional<IDilemmaMove> computeNextMove(final int gameId, final IDilemmaPlayer player, 
            final IDilemmaState state) 
            throws GameException, InterruptedException {
        final IGameMemory<DilemmaRoundData> memory = getMemoryForPlayer(player, state);
        final IDilemmaMoveFactory moveFactory = getMoveFactory();
        if (memory.size() == 0) { // cooperate for first round
            return Optional.of(moveFactory.createCooperateMove());
        }

        // imitate opponents move every other round thereafter
        final DilemmaRoundData previousRound = memory.getRound(0, true);
        final IDilemmaMove otherPlayersAction = previousRound.forOpponentOf(player).move();
        if (otherPlayersAction.equals(getMoveFactory().createCooperateMove())) {
            return Optional.of(otherPlayersAction);
        } else {
            final Map<DilemmaAnswerType, Map<DilemmaAnswerType, Double>> outcomes = player.getPossibleOutcomes();
            // Get payouts from outcomes map. Payouts are statically defined, such that we can access
            // them here correctly and calculate the probability of cooperation.
            final double varR = outcomes.get(DilemmaAnswerType.COOPERATE).get(DilemmaAnswerType.COOPERATE); // reward
            final double varS = outcomes.get(DilemmaAnswerType.COOPERATE).get(DilemmaAnswerType.DEFECT); // sucker
            final double varT = outcomes.get(DilemmaAnswerType.DEFECT).get(DilemmaAnswerType.COOPERATE); // temptation
            final double varP = outcomes.get(DilemmaAnswerType.DEFECT).get(DilemmaAnswerType.DEFECT); // punishment

            final double num1 = 1 - ((varT - varR) / (varR - varS));
            final double num2 = (varR - varP) / (varT - varP);
            // probability with which to cooperate, if enemy has defected in the previous round
            final double probabilityForCooperation = Math.min(num1, num2);

            return Optional.of(RANDOM.nextDouble() <= probabilityForCooperation
                ? getMoveFactory().createCooperateMove()
                : getMoveFactory().createDefectMove());
        }
    }
}