package de.fhdw.gaming.ipspiel23.dilemma.strategy.internals.gradual_tit_for_tat;

import java.util.Optional;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.ipspiel23.dilemma.domain.IDilemmaPlayer;
import de.fhdw.gaming.ipspiel23.dilemma.domain.IDilemmaState;
import de.fhdw.gaming.ipspiel23.dilemma.moves.IDilemmaMove;
import de.fhdw.gaming.ipspiel23.dilemma.moves.IDilemmaMoveFactory;
import de.fhdw.gaming.ipspiel23.dilemma.strategy.internals.DilemmaMemoryStrategy;
import de.fhdw.gaming.ipspiel23.dilemma.strategy.internals.DilemmaRoundData;
import de.fhdw.gaming.ipspiel23.dilemma.strategy.internals.DilemmaRoundPlayerData;
import de.fhdw.gaming.ipspiel23.memory.GameMemoryCapacity;
import de.fhdw.gaming.ipspiel23.memory.IGameMemory;
import de.fhdw.gaming.ipspiel23.memory.IGameMemoryCapacity;

import static de.fhdw.gaming.ipspiel23.dilemma.domain.DilemmaAnswerType.COOPERATE;
import static de.fhdw.gaming.ipspiel23.dilemma.domain.DilemmaAnswerType.DEFECT;

/**
 * TFT with two differences: 
 * <p>
 * (1) it increases the string of punishing defection responses with each additional defection by its opponent 
 * </p>
 * <p>
 * (2) it apologizes for each string of defections by cooperating in the subsequent two rounds.
 * </p>
 */
public class DilemmaGradualTitForTatStrategy extends DilemmaMemoryStrategy {

    /**
     * Creates a new instance of the {@link DilemmaGradualTitForTatStrategy} using the specified move factory.
     * 
     * @param moveFactory the move factory to use
     */
    protected DilemmaGradualTitForTatStrategy(final IDilemmaMoveFactory moveFactory) {
        super(moveFactory);
    }

    @Override
    public Optional<IDilemmaMove> computeNextMove(final int gameId, final IDilemmaPlayer player, 
            final IDilemmaState state)
            throws GameException, InterruptedException {
        final IGameMemory<DilemmaRoundData> memory = getMemoryForPlayer(player, state);
        final IDilemmaMoveFactory moveFactory = getMoveFactory();
        // cooperate on the first round ...
        if (memory.size() == 0) {
            return Optional.of(moveFactory.createCooperateMove());
        }
        final DilemmaRoundData previousRound = memory.getRound(0, true);
        final DilemmaRoundPlayerData myPreviousAction = previousRound.forPlayer(player);
        final DilemmaRoundPlayerData otherPlayersAction = previousRound.forOpponentOf(player);
        // what happened the round before?
        // did I cooperate?
        if (myPreviousAction.answer().equals(COOPERATE)) {
            // Am I currently apologizing?
            if (memory.size() > 1 && memory.getRound(1, true).forPlayer(player).answer().equals(DEFECT)) {
                // yes. cooperate again
                return Optional.of(moveFactory.createCooperateMove());
            }
            // no. What did the opponent do previously?
            if (otherPlayersAction.answer().equals(COOPERATE)) {
                // they were nice to us, so we do the same
                return Optional.of(moveFactory.createCooperateMove());
            }
            // they betrayed us :C 
            // let the punishment begin
            return Optional.of(moveFactory.createDefectMove());
        }
        // I was previously punishing my opponent!
        // do I need to continue?
        if (continuePunishingOpponent(player, memory)) {
            // yes. make them suffer...
            return Optional.of(moveFactory.createDefectMove());
        }
        // no we're even now. Let's apologize
        return Optional.of(moveFactory.createCooperateMove());
    }
    
    /**
     * determines whether to continue punishing the opponent of the specified player.
     * @param player the player
     * @param memory the memory of the player.
     * @return true if we need to continue the punishment, otherwise false.
     */
    private static boolean continuePunishingOpponent(final IDilemmaPlayer player, 
            final IGameMemory<DilemmaRoundData> memory) {
        // I was previously punishing my opponent!
        // how much did I punish them already?
        int punishCounter;
        for (punishCounter = 0; punishCounter < memory.size() 
            && memory.getRound(punishCounter, true)
                .forPlayer(player)
                .answer().equals(DEFECT);) {
            // more PMD-induced disgustingness :P
            punishCounter++;
        }
        // how often did I plan on punishing them?
        int defectsBeforePunishment = 0;
        // we don't count defects since we started the punishment to prevent deadlock.
        for (int i = punishCounter; i < memory.size(); i++) {
            if (memory.getRound(i, true).forOpponentOf(player).answer().equals(DEFECT)) {
                defectsBeforePunishment++;
            }
        }
        return punishCounter < defectsBeforePunishment;
    }
    
    @Override
    protected IGameMemoryCapacity requestedMemoryCapacity() {
        // perfect memory would be good
        return GameMemoryCapacity.unlimited();
    }
}