package de.fhdw.gaming.ipspiel23.dilemma.strategy.internals.grim;

import java.util.Optional;

import de.fhdw.gaming.ipspiel23.dilemma.domain.DilemmaAnswerType;
import de.fhdw.gaming.ipspiel23.dilemma.domain.IDilemmaPlayer;
import de.fhdw.gaming.ipspiel23.dilemma.domain.IDilemmaState;
import de.fhdw.gaming.ipspiel23.dilemma.domain.IDilemmaStrategy;
import de.fhdw.gaming.ipspiel23.dilemma.moves.IDilemmaMove;
import de.fhdw.gaming.ipspiel23.dilemma.moves.IDilemmaMoveFactory;
import de.fhdw.gaming.ipspiel23.dilemma.strategy.internals.DilemmaMemoryStrategy;
import de.fhdw.gaming.ipspiel23.dilemma.strategy.internals.DilemmaRoundData;
import de.fhdw.gaming.ipspiel23.dilemma.strategy.internals.DilemmaRoundPlayerData;
import de.fhdw.gaming.ipspiel23.memory.GameMemoryCapacity;
import de.fhdw.gaming.ipspiel23.memory.IGameMemory;
import de.fhdw.gaming.ipspiel23.memory.IGameMemoryCapacity;

/**
 * Implements {@link IDilemmaStrategy} by cooperating until the enemy defected once, 
 * and then defecting for the rest of the game.
 */
public class DilemmaGrimStrategy extends DilemmaMemoryStrategy {

    /**
     * Creates an {@link DilemmaGrimStrategy}.
     *
     * @param moveFactory The factory for creating Dilemma moves.
     */
    DilemmaGrimStrategy(final IDilemmaMoveFactory moveFactory) {
        super(moveFactory);
    }
    
    @Override
    public Optional<IDilemmaMove> computeNextMove(final int gameId,
            final IDilemmaPlayer player,
            final IDilemmaState state) {
        final IGameMemory<DilemmaRoundData> memory = getMemoryForPlayer(player, state);
        // always cooperate on first round
        if (memory.size() == 0) {            
            return Optional.of(getMoveFactory().createCooperateMove());
        }
        // check if enemy defected in any of the last rounds
        for (int i = 0; i < memory.size(); i++) {
            final DilemmaRoundData previousRound = memory.getRound(i, true);
            final DilemmaRoundPlayerData otherPlayersAction = previousRound.forOpponentOf(player);
            if (otherPlayersAction.answer().equals(DilemmaAnswerType.DEFECT)) {
                return Optional.of(getMoveFactory().createDefectMove());
            }
        }
        
        return Optional.of(getMoveFactory().createCooperateMove());
    }

    @Override
    protected IGameMemoryCapacity requestedMemoryCapacity() {
        return GameMemoryCapacity.unlimited();
    }
}
