package de.fhdw.gaming.ipspiel23.dilemma.strategy.internals;

import de.fhdw.gaming.ipspiel23.dilemma.domain.IDilemmaPlayer;
import de.fhdw.gaming.ipspiel23.dilemma.domain.IDilemmaState;
import de.fhdw.gaming.ipspiel23.dilemma.moves.IDilemmaMoveFactory;
import de.fhdw.gaming.ipspiel23.memory.GameMemoryIdentifier;
import de.fhdw.gaming.ipspiel23.memory.IGameMemory;
import de.fhdw.gaming.ipspiel23.memory.IGameMemoryCapacity;
import de.fhdw.gaming.ipspiel23.memory.IGameMemoryIdentifier;
import de.fhdw.gaming.ipspiel23.memory.IGameMemoryProvider;

/**
 * Represents the base class for all strategies that require a memory to access 
 * the decisions of previous rounds against the same opponent.
 */
public abstract class DilemmaMemoryStrategy extends DilemmaStrategy {

    /**
     * Creates a new instance of the {@link DilemmaMemoryStrategy} class.
     * @param moveFactory The move factory to use for creating moves.
     */
    protected DilemmaMemoryStrategy(final IDilemmaMoveFactory moveFactory) {
        super(moveFactory);
    }

    /**
     * How good should the memory be? (how many rounds in the past can be remembered?)
     */
    protected abstract IGameMemoryCapacity requestedMemoryCapacity();

    /**
     * Gets or creates the memory for the given player - strategy combination.
     * if the memory does not exist yet, it will be created.
     * @param player The player to get the memory for.
     * @param state The current state of the game.
     * @return The memory for the given player - strategy combination.
     */
    protected final IGameMemory<DilemmaRoundData> getMemoryForPlayer(final IDilemmaPlayer player, 
            final IDilemmaState state) {
        final IGameMemoryProvider memoryProvider = state.getMemoryProvider();
        final IDilemmaPlayer opponent = state.getOpponentOf(player);
        final IGameMemoryIdentifier identifier = GameMemoryIdentifier
            .of(player, player.getStrategy(), opponent.getStrategy());
        return memoryProvider.requestMemoryForStrategy(identifier, requestedMemoryCapacity());
    }

    @Override
    public int hashCode() {
        int hash = 17;
        hash = hash * 31 + toString().hashCode();
        return hash;
    }

    @Override 
    public String toString() {
        return getClass().getSimpleName();
    }

    @Override 
    public boolean equals(final Object other) {
        return other != null 
            && other.getClass().equals(getClass())
            && other.hashCode() == hashCode();
    }
}
