package de.fhdw.gaming.ipspiel23.dilemma.strategy.internals.reactive;

import java.util.Optional;
import java.util.Random;

import de.fhdw.gaming.ipspiel23.dilemma.domain.DilemmaAnswerType;
import de.fhdw.gaming.ipspiel23.dilemma.domain.IDilemmaPlayer;
import de.fhdw.gaming.ipspiel23.dilemma.domain.IDilemmaState;
import de.fhdw.gaming.ipspiel23.dilemma.domain.IDilemmaStrategy;
import de.fhdw.gaming.ipspiel23.dilemma.moves.IDilemmaMove;
import de.fhdw.gaming.ipspiel23.dilemma.moves.IDilemmaMoveFactory;
import de.fhdw.gaming.ipspiel23.dilemma.strategy.internals.DilemmaMemoryStrategy;
import de.fhdw.gaming.ipspiel23.dilemma.strategy.internals.DilemmaRoundData;
import de.fhdw.gaming.ipspiel23.dilemma.strategy.internals.DilemmaRoundPlayerData;
import de.fhdw.gaming.ipspiel23.memory.GameMemoryCapacity;
import de.fhdw.gaming.ipspiel23.memory.IGameMemory;
import de.fhdw.gaming.ipspiel23.memory.IGameMemoryCapacity;

/**
 * Implements {@link IDilemmaStrategy} by cooperating with probability y in first round 
 * and with probabilities p or q after opponent cooperates or defects.
 */
public class DilemmaReactiveStrategy extends DilemmaMemoryStrategy {
    
    /**
     * Probability to cooperate on the first round.
     */
    static final double COOPERATION_FIRST_ROUND_PROB = 0.5d;
    
    /**
     * Probability to cooperate after the opponent cooperated.
     */
    static final double COOPERATION_AFTER_COOPERATION_PROB = 0.8d;
    
    /**
     * Probability to cooperate after the opponent defected.
     */
    static final double COOPERATION_AFTER_DEFECT_PROB = 0.2d;
    
    /**
     * A random number generator to implement playing moves based on probabilities.
     */
    private static final Random RANDOM = new Random();

    /**
     * Creates an {@link DilemmaReactiveStrategy}.
     *
     * @param moveFactory The factory for creating Dilemma moves.
     */
    DilemmaReactiveStrategy(final IDilemmaMoveFactory moveFactory) {
        super(moveFactory);
    }
    
    @Override
    public Optional<IDilemmaMove> computeNextMove(final int gameId,
            final IDilemmaPlayer player,
            final IDilemmaState state) {
        final IGameMemory<DilemmaRoundData> memory = getMemoryForPlayer(player, state);
        final IDilemmaMoveFactory moveFactory = getMoveFactory();
        // cooperate on first round with a probability of y
        if (memory.size() == 0) {
            if (RANDOM.nextDouble() < COOPERATION_FIRST_ROUND_PROB) {
                return Optional.of(moveFactory.createCooperateMove());
            }
            return Optional.of(moveFactory.createDefectMove());
        }
        // cooperate with probability of p if opponent cooperated last round
        final DilemmaRoundData previousRound = memory.getRound(0, true);
        final DilemmaRoundPlayerData otherPlayersAction = previousRound.forOpponentOf(player);
        if (otherPlayersAction.answer().equals(DilemmaAnswerType.COOPERATE)) {
            if (RANDOM.nextDouble() < COOPERATION_AFTER_COOPERATION_PROB) {                
                return Optional.of(moveFactory.createCooperateMove());
            }
            return Optional.of(moveFactory.createDefectMove());
        }
        // cooperate with probability of y if opponent defected last round
        if (RANDOM.nextDouble() < COOPERATION_AFTER_DEFECT_PROB) {
            return Optional.of(moveFactory.createCooperateMove());
        }
        return Optional.of(moveFactory.createDefectMove());
    }

    @Override
    protected IGameMemoryCapacity requestedMemoryCapacity() {
        return GameMemoryCapacity.unlimited();
    }
}
