package de.fhdw.gaming.ipspiel23.dilemma.domain.internals;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.function.Consumer;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.ipspiel23.dilemma.domain.DilemmaAnswerType;
import de.fhdw.gaming.ipspiel23.dilemma.domain.IDilemmaPlayer;
import de.fhdw.gaming.ipspiel23.dilemma.domain.IDilemmaPlayerBuilder;
import de.fhdw.gaming.ipspiel23.dilemma.domain.IDilemmaStrategy;

/**
 * Implements {@link IDilemmaPlayerBuilder}.
 */
public final class DilemmaPlayerBuilder implements IDilemmaPlayerBuilder {

    /**
     * The name of the player.
     */
    private Optional<String> name;
    /**
     * The possible outcomes of this player. The key for the first-level map is the answer of the first player, the key
     * for the second-level map is the answer of the second player.
     */
    private Optional<Map<DilemmaAnswerType, Map<DilemmaAnswerType, Double>>> possibleOutcomes;

    /**
     * The hook that is used to inject the strategy into the player object when the player is added to the game.
     */
    private final Map<IDilemmaPlayer, Consumer<IDilemmaStrategy>> playerStrategyHooks; 

    /**
     * The parent game builder. Used to to register new players via callback.
     */
    private final DilemmaGameBuilder gameBuilder;

    /**
     * Creates an {@link DilemmaPlayerBuilder}.
     * @param parentBuilder the player builder that created this player instance.
     */
    public DilemmaPlayerBuilder(final DilemmaGameBuilder parentBuilder) {
        this.gameBuilder = parentBuilder;
        this.name = Optional.empty();
        this.possibleOutcomes = Optional.empty();
        this.playerStrategyHooks = new HashMap<>();
    }

    @Override
    public DilemmaPlayerBuilder changeName(final String newName) {
        this.name = Optional.of(newName);
        return this;
    }

    @Override
    public DilemmaPlayerBuilder changePossibleOutcomes(
            final Map<DilemmaAnswerType, Map<DilemmaAnswerType, Double>> newPossibleOutcomes) {
        this.possibleOutcomes = Optional.ofNullable(newPossibleOutcomes);
        return this;
    }

    @Override
    public IDilemmaPlayer build() throws GameException {
        final IDilemmaPlayer player = new DilemmaPlayer(
            this, 
            this.name.orElseThrow(),
            this.checkPossibleOutcomes(this.possibleOutcomes.orElseThrow()));
        gameBuilder.onPlayerBuilt(this, player);
        return player;
    }

    /**
     * Checks if all possible outcomes are defined for a player.
     *
     * @param outcomes The possible outcomes for the player.
     */
    private Map<DilemmaAnswerType, Map<DilemmaAnswerType, Double>> checkPossibleOutcomes(
            final Map<DilemmaAnswerType, Map<DilemmaAnswerType, Double>> outcomes) {
        this.checkPossibleOutcome(outcomes, DilemmaAnswerType.COOPERATE, DilemmaAnswerType.COOPERATE);
        this.checkPossibleOutcome(outcomes, DilemmaAnswerType.COOPERATE, DilemmaAnswerType.DEFECT);
        this.checkPossibleOutcome(outcomes, DilemmaAnswerType.DEFECT, DilemmaAnswerType.COOPERATE);
        this.checkPossibleOutcome(outcomes, DilemmaAnswerType.DEFECT, DilemmaAnswerType.DEFECT);
        return outcomes;
    }

    /**
     * Sets the hook that is used to inject the strategy into the player object when the player is added to the game.
     * 
     * @param player The player for which the hook is registered.
     * @param value The hook that is used to inject the strategy into the player object when the player is 
     * added to the game.
     */
    void registerPlayerStrategyHook(final IDilemmaPlayer player, final Consumer<IDilemmaStrategy> value) {
        this.playerStrategyHooks.put(player, value);
    }

    /**
     * Injects the strategy into the player object.
     * 
     * @param player The player object that is injected with the strategy.
     * @param strategy The strategy that is injected into the player object.
     */
    void injectPlayerStrategyUsingHook(final IDilemmaPlayer player, final IDilemmaStrategy strategy) {
        final Consumer<IDilemmaStrategy> playerStrategyHook = playerStrategyHooks.getOrDefault(player, null);
        if (playerStrategyHook == null) {
            throw new IllegalStateException("Attempted to inject strategy into player object before "
                + "hook was configured!");
        }
        // we don't want to expose the strategy setter to anyone, so it's done via this
        // hook kindly provided to us by the player constructor.
        playerStrategyHook.accept(strategy);
        playerStrategyHooks.remove(player);
    }

    /**
     * Checks if a given outcome is defined for a player.
     * Public for testing.
     *
     * @param outcomes     The possible outcomes for the player.
     * @param firstChoice  The choice of the first player.
     * @param secondChoice The choice of the second player.
     */
    void checkPossibleOutcome(
            final Map<DilemmaAnswerType, Map<DilemmaAnswerType, Double>> outcomes,
            final DilemmaAnswerType firstChoice,
            final DilemmaAnswerType secondChoice) throws IllegalArgumentException {
        if (outcomes == null || outcomes.getOrDefault(firstChoice, Collections.emptyMap()).get(secondChoice) == null) {
            throw new IllegalArgumentException(
                String.format("No outcome defined for player '%s' and combination %s/%s.",
                    this.name.orElse("Unnamed Player"),
                    firstChoice.toString(),
                    secondChoice.toString()));
        }
    }
}
