package de.fhdw.gaming.ipspiel23.dilemma.domain.internals;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import de.fhdw.gaming.core.domain.AbstractPlayer;
import de.fhdw.gaming.ipspiel23.dilemma.domain.DilemmaAnswerType;
import de.fhdw.gaming.ipspiel23.dilemma.domain.IDilemmaPlayer;
import de.fhdw.gaming.ipspiel23.dilemma.domain.IDilemmaStrategy;

/**
 * Implements {@link IDilemmaPlayer}.
 */
public final class DilemmaPlayer extends AbstractPlayer<IDilemmaPlayer> implements IDilemmaPlayer {
    
    /**
     * The answer of the player.
     */
    private Optional<DilemmaAnswerType> answer;

    /**
     * The strategy of the player.
     */
    private IDilemmaStrategy strategy;

    /**
     * The possible outcomes of this player. The key for the first-level map is the answer of the first player, the key
     * for the second-level map is the answer of the second player.
     */
    private final Map<DilemmaAnswerType, Map<DilemmaAnswerType, Double>> possibleOutcomes;

    /**
     * Creates a Dilemma player.
     *
     * @param builder          The parent player builder that is used to create this player.
     * @param name             The name of the player.
     * @param possibleOutcomes The possible outcomes of this player as DilemmaAnswerType.
     *                         The key for the first-level map is the answer of the first player,
     *                         the key for the second-level map is the answer of the second player.
     */
    DilemmaPlayer(final DilemmaPlayerBuilder builder, final String name,
            final Map<DilemmaAnswerType, Map<DilemmaAnswerType, Double>> possibleOutcomes) {
        super(name);
        this.possibleOutcomes = Collections
            .unmodifiableMap(new LinkedHashMap<>(Objects.requireNonNull(possibleOutcomes, "possibleOutcomes")));
        this.answer = Optional.empty();

        // Register the strategy hook
        // This hook is used to carefully expose the strategy setter to the builder
        // and nothing else.
        // the builder will then use the hook to inject the strategy into the player.
        builder.registerPlayerStrategyHook(this, newStrategy -> this.strategy = newStrategy);
    }

    /**
     * Creates a Dilemma player.
     *
     * @param source The {@link IDilemmaPlayer} to copy.
     */
    DilemmaPlayer(final IDilemmaPlayer source) {
        super(source);
        this.possibleOutcomes = source.getPossibleOutcomes();
        this.answer = source.getAnswer();
        this.strategy = source.getStrategy();
    }

    @Override
    public boolean equals(final Object object) {
        if (object instanceof DilemmaPlayer) {
            final DilemmaPlayer other = (DilemmaPlayer) object;
            return super.equals(object) && this.answer.equals(other.answer)
                && this.possibleOutcomes.equals(other.possibleOutcomes);
        }
        return false;
    }

    @Override
    public int hashCode() {
        return Objects.hash(super.hashCode(), answer, possibleOutcomes);
    }

    /**
     * The possible outcomes of this player. The key for the first-level map is the answer of the first player, the key
     * for the second-level map is the answer of the second player.
     */
    @Override
    public Map<DilemmaAnswerType, Map<DilemmaAnswerType, Double>> getPossibleOutcomes() {
        return this.possibleOutcomes;
    }

    @Override
    public Optional<DilemmaAnswerType> getAnswer() {
        return this.answer;
    }

    @Override
    public void setAnswer(final DilemmaAnswerType newAnswer) {
        if (this.answer.isPresent()) {
            throw new IllegalStateException(String.format("Player %s tried to change her answer.", this.getName()));
        }
        this.answer = Optional.of(newAnswer);
    }

    @Override
    public IDilemmaPlayer deepCopy() {
        return new DilemmaPlayer(this);
    }

    /**
     * Is called to provide the outcome of the game for the player.
     */
    @Override
    public String toString() {
        return String
            .format("DilemmaPlayer[name=%s, state=%s, outcome=%s, answer=%s]",
                this.getName(),
                this.getState(),
                this.getOutcome(),
                this.answer);
    }

    @Override
    public IDilemmaStrategy getStrategy() {
        if (this.strategy == null) {
            throw new IllegalStateException("The strategy of player " + this.getName() + " was never set.");
        }
        return this.strategy;
    }
}
