package de.fhdw.gaming.ipspiel23.c4.domain.impl;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import de.fhdw.gaming.ipspiel23.c4.domain.C4PositionMaterializer;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Board;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Position;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4BoardSlim;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Field;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Player;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Solution;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4SolutionSlim;

/**
 * The default implementation of {@link IC4Board}.
 */
public class C4Board implements IC4Board {

    /**
     * The maximum number of fields that are allowed to be cached.
     * For performance reasons, the field cache is limited to a certain size.
     * At some point, it becomes more expensive to cache the fields than to create them on the fly.
     * @hidden the value 2^16 was chosen arbitrarily, but should be fine for most use cases :)
     * We just want to avoid that the field cache grows indefinitely, consuming more and more memory.
     * that will never be freed again due to the strong references to the fields.
     */
    private static final int FIELD_CACHE_LIMIT = 1 << 16;

    /**
     * A lazily populated cache for the field instances of this board.
     */
    private final Map<IC4Position, IC4Field> fieldCache; 

    /**
     * The internal slim board that is used to store the actual board state.
     */
    private final IC4BoardSlim slimBoard;

    /**
     * A thread local buffer that is used to receive the dematerialized positions from the slim board.
     */
    private final ThreadLocal<long[]> localPositionBuffer;

    /**
     * Creates a new instance of {@link C4Board}.
     * 
     * @param slimBoard The internal slim board that is used to store the actual board state.
     */
    public C4Board(final IC4BoardSlim slimBoard) {
        this.slimBoard = slimBoard;
        this.fieldCache = new HashMap<>(slimBoard.getBoardStateUnsafe().length);
        this.localPositionBuffer = ThreadLocal.withInitial(() -> 
            new long[this.slimBoard.getBoardStateUnsafe().length]);
    }

    /**
     * Returns the field instance cache.
     */
    public Map<IC4Position, IC4Field> getFieldCache() {
        return this.fieldCache;
    }

    @Override
    public int getRowCount() {
        return this.slimBoard.getRowCount();
    }

    @Override
    public int getColumnCount() {
        return this.slimBoard.getColumnCount();
    }

    @Override
    public int getMinimumSolutionSize() {
        return this.slimBoard.getMinimumSolutionSize();
    }

    @Override
    public IC4Board deepCopy() {
        return new C4Board(this.slimBoard.deepCopy());
    }

    @Override
    public IC4BoardSlim getInternalBoard() {
        return this.slimBoard;
    }

    @Override
    public boolean checkBounds(final IC4Position position) {
        return this.checkBounds(position.getRow(), position.getColumn());
    }

    @Override
    public boolean checkBounds(final int row, final int column) {
        return this.slimBoard.checkBounds(row, column);
    }

    @Override
    public boolean isEmpty(final IC4Position position) {
        return this.isEmpty(position.getRow(), position.getColumn());
    }

    @Override
    public boolean isEmpty(final int row, final int column) {
        if (!this.checkBounds(row, column)) {
            throw new IndexOutOfBoundsException("The provided position is not within the defined bounds of the board!");
        }
        return this.slimBoard.isEmptyUnsafe(row, column);
    }

    @Override
    public Optional<IC4Field> tryGetField(final IC4Position position) {
        return Optional.ofNullable(this.getFieldInternal(position, false));
    }

    @Override
    public IC4Field getField(final IC4Position position) {
        return this.getFieldInternal(position, true);
    }

    /**
     * Retrieves the field from the specified position, either from cache or from the board itself.
     * @param position the position to retrieve the field from
     * @param throwOob whether an {@link IndexOutOfBoundsException} should be thrown if the bounds are violated.
     * @return the field or null if throwOob is false.
     */
    private IC4Field getFieldInternal(final IC4Position position, final boolean throwOob) {
        IC4Field field = this.fieldCache.get(position);
        if (field != null) {
            return field;
        }
        if (!this.checkBounds(position)) {
            if (throwOob) {
                throw new IndexOutOfBoundsException("The provided position violates the bounds of the board!");
            }
            return null;
        }
        field = new C4Field(this, position);
        // assume that later fields are more likely to be accessed again
        // therefore, clear the cache if it is full allowing the new field to be cached
        if (this.fieldCache.size() >= FIELD_CACHE_LIMIT) {
            this.fieldCache.clear();
        }
        this.fieldCache.put(position, field);
        return field;
    }

    @Override
    public Optional<IC4Player> getOccupyingPlayerOrDefault(final IC4Position position) {
        final int row = position.getRow();
        final int column = position.getColumn();
        if (!this.checkBounds(row, column)) {
            throw new IndexOutOfBoundsException("The provided position is not within the defined bounds of the board!");
        }
        final int token = this.slimBoard.getTokenUnsafe(row, column);
        return Optional.ofNullable(this.slimBoard.getPlayerByToken(token));
    }

    @Override
    public IC4Field[][] getFields() {
        // this is the GC's worst nightmare, but whatever :P
        // if you call this method, you probably don't care about performance anyway
        // no need for fancy caching here ¯\_(ツ)_/¯
        final int rows = this.getRowCount();
        final int cols = this.getColumnCount();
        final IC4Field[][] fields = new C4Field[rows][cols];
        for (int row = 0; row < rows; row++) {
            for (int col = 0; col < cols; col++) {
                final IC4Position position = new C4Position(row, col);
                fields[row][col] = this.getFieldInternal(position, false);
            }
        }
        return fields;
    }

    @Override
    public Optional<IC4Solution> tryFindFirstSolution() {
        final IC4SolutionSlim slimSolution = this.slimBoard.tryFindFirstSolution(true);
        if (slimSolution == null) {
            return Optional.empty();
        }
        final IC4Solution solution = new C4Solution(this, slimSolution);
        return Optional.of(solution);
    }

    @Override
    public Set<IC4Solution> findAllSolutions() {
        final Set<IC4SolutionSlim> slimSolutions = this.slimBoard.findAllSolutions(true);
        if (slimSolutions.isEmpty()) {
            return Set.of();
        }
        final Set<IC4Solution> solutions = new HashSet<>(slimSolutions.size());
        for (final IC4SolutionSlim slimSolution : slimSolutions) {
            solutions.add(new C4Solution(this, slimSolution));
        }
        return solutions;
    }

    @Override
    public boolean isSolid(final IC4Position position) {
        return this.isSolid(position.getRow(), position.getColumn());
    }

    @Override
    public boolean isSolid(final int row, final int column) {
        // we need to also shift the row by one to the top because 
        // the row below the board is also considered solid and
        // therefore for the context of this method, the board is
        // one row higher than it actually is
        if (!this.checkBounds(row, column) && !this.checkBounds(row - 1, column)) {
            throw new IndexOutOfBoundsException("The provided position is not within the defined bounds of the board!");
        }
        return this.slimBoard.isSolidUnsafe(row, column);
    }

    @Override
    public IC4Position[] getEmptyPositions() {
        return this.getPositionsByToken(this.slimBoard.emptyToken());
    }

    @Override
    public IC4Position[] getPositionsByPlayer(final IC4Player player) {
        return this.getPositionsByToken(player.getToken());
    }

    /**
     * Returns all positions on the board that match the given token.
     * @param token The token to search for.
     * @return All positions on the board that match the given token.
     */
    private IC4Position[] getPositionsByToken(final int token) {
        // re-use the same thread-local buffer :)
        final long[] buffer = this.localPositionBuffer.get();
        final int positionsRead = this.slimBoard.getDematPositionsByTokenUnsafe(buffer, token);
        final IC4Position[] positions = new IC4Position[positionsRead];
        for (int i = 0; i < positions.length && i < buffer.length; i++) {
            positions[i] = C4PositionMaterializer.rematerialize(buffer[i]);
        }
        return positions;
    }

    @Override
    public IC4Position[] getLegalPositions() {
        final long[] buffer = this.localPositionBuffer.get();
        final int positionsRead = this.slimBoard.getLegalDematPositionsUnsafe(buffer);
        final IC4Position[] positions = new IC4Position[positionsRead];
        for (int i = 0; i < positions.length && i < buffer.length; i++) {
            positions[i] = C4PositionMaterializer.rematerialize(buffer[i]);
        }
        return positions;
    }

    @Override
    public int countEmptyPositions() {
        return this.countPositionsByToken(this.slimBoard.emptyToken());
    }

    @Override
    public int countPositionsByPlayer(final IC4Player player) {
        return this.countPositionsByToken(player.getToken());
    }

    /**
     * Counts all positions on the board that match the given token.
     * @param token The token to search for.
     * @return The number of positions on the board that match the given token.
     */
    private int countPositionsByToken(final int token) {
        final long[] buffer = this.localPositionBuffer.get();
        return this.slimBoard.getDematPositionsByTokenUnsafe(buffer, token);
    }

    @Override
    public int countLegalPositions() {
        final long[] buffer = this.localPositionBuffer.get();
        return this.slimBoard.getLegalDematPositionsUnsafe(buffer);
    }

    @Override
    public int hashCode() {
        int hash = 7;
        final int[] board = this.slimBoard.getBoardStateUnsafe();
        for (final int token : board) {
            hash = hash * 31 + token;
        }
        hash = hash * 31 + this.getMinimumSolutionSize();
        hash = hash * 31 + this.getRowCount();
        hash = hash * 31 + this.getColumnCount();
        return hash;
    }
    
    @Override
    public boolean equals(final Object object) {
        if (object == this) {
            return true;
        }
        if (!(object instanceof IC4Board)) {
            return false;
        }
        final IC4Board other = (IC4Board) object;
        // quick check if dimensions even match
        if (this.getRowCount() != other.getRowCount() || this.getColumnCount() != other.getColumnCount()) {
            return false;
        }
        // more thorough check over every field
        final int[] myBoard = this.slimBoard.getBoardStateUnsafe();
        final int[] otherBoard = other.getInternalBoard().getBoardStateUnsafe();
        for (int i = 0; i < myBoard.length; i++) {
            if (myBoard[i] != otherBoard[i]) {
                return false;
            }
        }
        return this.getMinimumSolutionSize() == other.getMinimumSolutionSize();
    }

    @Override
    public boolean isFull() {
        return this.slimBoard.isFull();
    }

    @Override
    public String toString() {
        return "C4Board{"
            + "slimBoard=" + this.slimBoard
            + '}';
    }
}
