package de.fhdw.gaming.ipspiel23.freizeitgestaltung.domain.impl;

import java.util.Collections;
import java.util.Map;
import java.util.Optional;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.ipspiel23.freizeitgestaltung.domain.FzgPlayer;
import de.fhdw.gaming.ipspiel23.freizeitgestaltung.domain.FzgPlayerBuilder;
import de.fhdw.gaming.ipspiel23.freizeitgestaltung.move.AnswerOptions;

/**
 * implements {@link FzgPlayerBuilder}.
 * 
 */
public class FzgPlayerBuilderImpl implements FzgPlayerBuilder {

    /**
     * The name of the player.
     */
    private Optional<String> name;
    /**
     * The possible outcomes of this player. The key for the first-level map is the answer of the first player, the key
     * for the second-level map is the answer of the second player.
     */
    private Optional<Map<AnswerOptions, Map<AnswerOptions, Double>>> possibleOutcomes;

    /**
     * Constructor.
     */
    public FzgPlayerBuilderImpl() {
        this.name = Optional.empty();
        this.possibleOutcomes = Optional.empty();
    }

    @Override
    public FzgPlayerBuilder changeName(final String newName) {
        this.name = Optional.of(newName);
        return this;
    }

    @Override
    public FzgPlayerBuilder changePossibleOutcomes(
            final Map<AnswerOptions, Map<AnswerOptions, Double>> newPossibleOutcomes) {
        this.possibleOutcomes = Optional.of(newPossibleOutcomes);
        return this;
    }

    @Override
    public FzgPlayer build() throws GameException {
        return new FzgPlayerImpl(this.name.orElseThrow(), checkPossibleOutcomes(this.possibleOutcomes.orElseThrow()));
    }

    /**
     * checks if all possible outcomes are defined for the player.
     * 
     * @param outcomes The possible outcomes for the player.
     */
    private Map<AnswerOptions, Map<AnswerOptions, Double>> checkPossibleOutcomes(
            final Map<AnswerOptions, Map<AnswerOptions, Double>> outcomes) {
        this.checkPossibleOutcome(outcomes, AnswerOptions.CINEMA, AnswerOptions.CINEMA);
        this.checkPossibleOutcome(outcomes, AnswerOptions.CINEMA, AnswerOptions.FOOTBALL);
        this.checkPossibleOutcome(outcomes, AnswerOptions.FOOTBALL, AnswerOptions.CINEMA);
        this.checkPossibleOutcome(outcomes, AnswerOptions.FOOTBALL, AnswerOptions.FOOTBALL);
        return outcomes;
    }

    /**
     * Checks if a given outcome is defined for a player.
     *
     * @param outcomes     The possible outcomes for the player.
     * @param firstChoice  The choice of the first player.
     * @param secondChoice The choice of the second player.
     */
    private void checkPossibleOutcome(final Map<AnswerOptions, Map<AnswerOptions, Double>> outcomes,
            final AnswerOptions firstChoice,
            final AnswerOptions secondChoice) {
        if (outcomes.getOrDefault(firstChoice, Collections.emptyMap()).get(secondChoice) == null) {
            throw new IllegalArgumentException(
                String.format(
                    "No outcome defined for player '%s' and combination %s/%s.",
                    this.name.get(),
                    firstChoice,
                    secondChoice));
        }
    }
}
