package de.fhdw.gaming.ipspiel23.c4.domain.impl;

import java.util.Optional;

import de.fhdw.gaming.ipspiel23.c4.domain.C4Direction;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Board;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Position;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Field;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Player;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Solution;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4SolutionSlim;

/**
 * The default implementation of {@link IC4Solution}.
 */
public class C4Solution implements IC4Solution {

    /**
     * The parent board that contains this solution.
     */
    private final IC4Board board;

    /**
     * The internal slim solution instance that is wrapped by this class.
     */
    private final IC4SolutionSlim slimSolution;

    /**
     * Creates a new instance.
     * 
     * @param board The parent board that contains this solution.
     * @param slimSolution The internal slim solution instance that is wrapped by this class.
     */
    public C4Solution(final IC4Board board, final IC4SolutionSlim slimSolution) {
        this.board = board;
        this.slimSolution = slimSolution;
    }

    @Override
    public IC4Player getOwner() {
        return this.slimSolution.getOwner();
    }

    @Override
    public IC4Position getStartPosition() {
        return this.slimSolution.getStartPosition();
    }

    @Override
    public IC4Position getEndPosition() {
        return this.slimSolution.getEndPosition();
    }

    @Override
    public C4Direction getDirection() {
        return this.slimSolution.getDirection();
    }

    @Override
    public int size() {
        return this.slimSolution.size();
    }

    @Override
    public IC4Field[] getContainingFields() {
        final IC4Field[] fields = new IC4Field[size()];
        final C4Direction dir = getDirection();
        // size() will always be > 0 :)
        Optional<IC4Field> field = this.board.tryGetField(getStartPosition());
        // add fields until we reach the end of the solution, there are no neighboring fields, or 
        // we reach an empty field
        for (int i = 0; i < fields.length && field.isPresent(); field = field.get().tryGetNeighbor(dir), i++) {
            fields[i] = field.get();
        }
        return fields;
    }

    @Override
    public boolean equals(final Object object) {
        if (object == this) {
            return true;
        }
        if (!(object instanceof IC4Solution)) {
            return false;
        }
        final IC4Solution other = (IC4Solution) object;
        return this.getOwner().getToken() == other.getOwner().getToken() 
                // we don't care what direction a solution is facing.
                // the direction itself should probably be internal, but I guess
                // it may have valid use cases for consumers of our library :)
                && (this.getStartPosition().equals(other.getStartPosition()) 
                        && this.getEndPosition().equals(other.getEndPosition())
                    || this.getStartPosition().equals(other.getEndPosition()) 
                        && this.getEndPosition().equals(other.getStartPosition()))
                && this.size() == other.size();
    }

    @Override
    public int hashCode() {
        int hash = 17 * 31;
        hash += this.board.hashCode();
        hash = hash * 31 + this.slimSolution.hashCode();
        return hash;
    }

    @Override
    public String toString() {
        return String.format("C4Solution[owner: %s, start: %s, end: %s, direction: %s]",
            getOwner(), getStartPosition(), getEndPosition(), getDirection());
    }
}
