/*
 * Copyright © 2021-2023 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel24-VierConnects-gui.
 *
 * ipspiel24-VierConnects-gui is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * ipspiel24-VierConnects-gui is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * ipspiel24-VierConnects-gui. If not, see <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel24.VierConnects.gui.impl;

import java.lang.ref.Reference;
import java.lang.ref.WeakReference;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicReference;

import de.fhdw.gaming.core.domain.Game;
import de.fhdw.gaming.gui.GuiObserver;
import de.fhdw.gaming.ipspiel24.VierConnects.core.domain.VierConnectsPlayer;
import de.fhdw.gaming.ipspiel24.VierConnects.core.domain.VierConnectsState;
import de.fhdw.gaming.ipspiel24.VierConnects.core.domain.VierConnectsStrategy;
import de.fhdw.gaming.ipspiel24.VierConnects.core.moves.VierConnectsMove;
import de.fhdw.gaming.ipspiel24.VierConnects.gui.VierConnectsBoardEventProvider;
import javafx.scene.Node;

/**
 * Implements {@link GuiObserver}.
 */
final class VierConnectsGuiObserverImpl implements GuiObserver {

    /**
     * The primary (i.e. first created) {@link VierConnectsGuiObserverImpl} instance of this class.
     */
    private static final AtomicReference<WeakReference<VierConnectsGuiObserverImpl>> INSTANCE = new AtomicReference<>();

    /**
     * The {@link VierConnectsBoardView} objects per Vier Connects game ID.
     */
    private final Map<Integer, VierConnectsBoardView> boardViews;

    /**
     * Creates an {@link VierConnectsGuiObserverImpl}.
     */
    VierConnectsGuiObserverImpl() {
        VierConnectsGuiObserverImpl.INSTANCE.compareAndSet(null, new WeakReference<>(this));
        this.boardViews = new LinkedHashMap<>();
    }

    /**
     * Returns a {@link VierConnectsBoardEventProvider} for a given game.
     *
     * @param gameId The game ID.
     * @return The {@link VierConnectsBoardEventProvider}.
     */
    static Optional<VierConnectsBoardEventProvider> getEventProvider(final int gameId) {
        final VierConnectsGuiObserverImpl instance = Optional.ofNullable(VierConnectsGuiObserverImpl.INSTANCE.get())
                .map(Reference::get).orElse(null);
        if (instance == null) {
            return Optional.empty();
        }
        final VierConnectsBoardView boardView = instance.boardViews.get(gameId);
        return boardView == null ? Optional.empty() : Optional.of(new VierConnectsBoardEventProviderImpl(boardView));
    }

    @Override
    public Optional<Node> gameCreated(final Game<?, ?, ?, ?> game) {
        final Optional<Game<VierConnectsPlayer, VierConnectsState, VierConnectsMove,
                VierConnectsStrategy>> vierConnectsGame = toVierConnectsGame(game);
        if (vierConnectsGame.isPresent()) {
            final VierConnectsBoardView boardView = new VierConnectsBoardView(vierConnectsGame.get());
            this.boardViews.put(game.getId(), boardView);
            return Optional.of(boardView.getNode());
        }
        return Optional.empty();
    }

    @Override
    public void gamePaused(final Game<?, ?, ?, ?> game) {
        final Optional<Game<VierConnectsPlayer, VierConnectsState, VierConnectsMove,
                VierConnectsStrategy>> vierConnectsGame = toVierConnectsGame(game);
        if (vierConnectsGame.isPresent()) {
            final VierConnectsBoardView boardView = this.boardViews.get(game.getId());
            if (boardView != null) {
                boardView.gamePaused(vierConnectsGame.get());
            }
        }
    }

    @Override
    public void gameResumed(final Game<?, ?, ?, ?> game) {
        final Optional<Game<VierConnectsPlayer, VierConnectsState, VierConnectsMove,
                VierConnectsStrategy>> vierConnectsGame = toVierConnectsGame(game);
        if (vierConnectsGame.isPresent()) {
            final VierConnectsBoardView boardView = this.boardViews.get(game.getId());
            if (boardView != null) {
                boardView.gameResumed(vierConnectsGame.get());
            }
        }
    }

    @Override
    public void gameDestroyed(final Game<?, ?, ?, ?> game) {
        final Optional<Game<VierConnectsPlayer, VierConnectsState, VierConnectsMove,
                VierConnectsStrategy>> vierConnectsGame = toVierConnectsGame(game);
        if (vierConnectsGame.isPresent()) {
            final VierConnectsBoardView boardView = this.boardViews.remove(game.getId());
            if (boardView != null) {
                boardView.destroy(vierConnectsGame.get());
            }
        }
    }

    /**
     * Casts the game to the right type if it is a Vier Connects game. Otherwise an empty {@link Optional} is returned.
     *
     * @param game The game to check.
     */
    private static Optional<
            Game<VierConnectsPlayer, VierConnectsState, VierConnectsMove, VierConnectsStrategy>> toVierConnectsGame(
            final Game<?, ?, ?, ?> game) {
        if (game.getState() instanceof VierConnectsState) {
            @SuppressWarnings("unchecked")
            final Game<VierConnectsPlayer, VierConnectsState, VierConnectsMove,
                    VierConnectsStrategy> vierConnectsGame = (Game<
                    VierConnectsPlayer, VierConnectsState, VierConnectsMove, VierConnectsStrategy>) game;
            return Optional.of(vierConnectsGame);
        }
        return Optional.empty();
    }
}
