
package de.fhdw.gaming.ipspiel24.fg.domain.impl;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import de.fhdw.gaming.core.domain.DefaultGame;
import de.fhdw.gaming.core.domain.Game;
import de.fhdw.gaming.core.domain.GameBuilder;
import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.Observer;
import de.fhdw.gaming.ipspiel24.fg.domain.FGGameBuilder;
import de.fhdw.gaming.ipspiel24.fg.domain.FGPlayer;
import de.fhdw.gaming.ipspiel24.fg.domain.FGPlayerBuilder;
import de.fhdw.gaming.ipspiel24.fg.domain.FGState;
import de.fhdw.gaming.ipspiel24.fg.domain.FGStrategy;
import de.fhdw.gaming.ipspiel24.fg.moves.FGMove;
import de.fhdw.gaming.ipspiel24.fg.moves.impl.AbstractFGMove;

/**
 * Implements {@link FGGameBuilder}.
 */
final class FGGameBuilderImpl implements FGGameBuilder {

    /**
     * The {@link Observer}s to be attached to the game.
     */
    private final List<Observer> observers;
    /**
     * The player using black tokens.
     */
    private Optional<FGPlayer> firstPlayer;
    /**
     * The strategy of the player using black tokens.
     */
    private Optional<FGStrategy> firstPlayerStrategy;
    /**
     * The player using white tokens.
     */
    private Optional<FGPlayer> secondPlayer;
    /**
     * The strategy of the player using white tokens.
     */
    private Optional<FGStrategy> secondPlayerStrategy;
    /**
     * The maximum computation time per move in seconds.
     */
    private int maxComputationTimePerMove;

    /**
     * Creates a FG game builder.
     */
    FGGameBuilderImpl() {
        this.observers = new ArrayList<>();
        this.firstPlayer = Optional.empty();
        this.firstPlayerStrategy = Optional.empty();
        this.secondPlayer = Optional.empty();
        this.secondPlayerStrategy = Optional.empty();
        this.maxComputationTimePerMove = GameBuilder.DEFAULT_MAX_COMPUTATION_TIME_PER_MOVE;
    }

    @Override
    public FGPlayerBuilder createPlayerBuilder() {
        return new FGPlayerBuilderImpl();
    }

    @Override
    public FGGameBuilder addPlayer(final FGPlayer player, final FGStrategy strategy)
            throws GameException {

        if (this.firstPlayer.isEmpty()) {
            this.firstPlayer = Optional.of(Objects.requireNonNull(player, "player"));
            this.firstPlayerStrategy = Optional.of(Objects.requireNonNull(strategy, "firstPlayerStrategy"));
        } else if (this.secondPlayer.isEmpty()) {
            this.secondPlayer = Optional.of(Objects.requireNonNull(player, "player"));
            this.secondPlayerStrategy = Optional.of(Objects.requireNonNull(strategy, "secondPlayerStrategy"));
        } else {
            throw new GameException(String.format("More than two players are now allowed."));
        }
        return this;
    }

    @Override
    public FGGameBuilder changeMaximumComputationTimePerMove(final int newMaxComputationTimePerMove) {
        this.maxComputationTimePerMove = newMaxComputationTimePerMove;
        return this;
    }

    @Override
    public FGGameBuilder addObservers(final List<Observer> newObservers) {
        this.observers.addAll(newObservers);
        return this;
    }

    @Override
    public Game<FGPlayer, FGState, FGMove, FGStrategy> build(final int id)
            throws GameException, InterruptedException {
        if (!this.firstPlayer.isPresent() || !this.secondPlayer.isPresent()) {
            throw new GameException("A FG game needs two players.");
        }

        final FGStateImpl initialState = new FGStateImpl(this.firstPlayer.get(), this.secondPlayer.get());

        final Map<String, FGStrategy> strategies = new LinkedHashMap<>();
        strategies.put(initialState.getFirstPlayer().getName(), this.firstPlayerStrategy.orElseThrow());
        strategies.put(initialState.getSecondPlayer().getName(), this.secondPlayerStrategy.orElseThrow());
        return new DefaultGame<>(
                id,
                initialState,
                strategies,
                this.maxComputationTimePerMove,
                AbstractFGMove.class::isInstance,
                new FGMoveGeneratorImpl(),
                this.observers);
    }
}
