/*
 * Copyright © 2020-2023 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of othello-gui.
 *
 * Othello-gui is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * Othello-gui is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * othello-gui. If not, see <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.othello.gui.impl;

import java.lang.ref.Reference;
import java.lang.ref.WeakReference;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicReference;

import de.fhdw.gaming.core.domain.Game;
import de.fhdw.gaming.gui.GuiObserver;
import de.fhdw.gaming.othello.core.domain.OthelloPlayer;
import de.fhdw.gaming.othello.core.domain.OthelloState;
import de.fhdw.gaming.othello.core.domain.OthelloStrategy;
import de.fhdw.gaming.othello.core.moves.OthelloMove;
import de.fhdw.gaming.othello.gui.OthelloBoardEventProvider;
import javafx.scene.Node;

/**
 * Implements {@link GuiObserver}.
 */
final class OthelloGuiObserverImpl implements GuiObserver {

    /**
     * The primary (i.e. first created) {@link OthelloGuiObserverImpl} instance of this class.
     */
    private static final AtomicReference<WeakReference<OthelloGuiObserverImpl>> INSTANCE = new AtomicReference<>();

    /**
     * The {@link OthelloBoardView} objects per Othello game ID.
     */
    private final Map<Integer, OthelloBoardView> boardViews;

    /**
     * Creates an {@link OthelloGuiObserverImpl}.
     */
    OthelloGuiObserverImpl() {
        OthelloGuiObserverImpl.INSTANCE.compareAndSet(null, new WeakReference<>(this));
        this.boardViews = new LinkedHashMap<>();
    }

    /**
     * Returns an {@link OthelloBoardEventProvider} for a given game.
     *
     * @param gameId The game ID.
     * @return The {@link OthelloBoardEventProvider}.
     */
    static Optional<OthelloBoardEventProvider> getEventProvider(final int gameId) {
        final OthelloGuiObserverImpl instance = Optional.ofNullable(OthelloGuiObserverImpl.INSTANCE.get())
                .map(Reference::get).orElse(null);
        if (instance == null) {
            return Optional.empty();
        }
        final OthelloBoardView boardView = instance.boardViews.get(gameId);
        return boardView == null ? Optional.empty() : Optional.of(new OthelloBoardEventProviderImpl(boardView));
    }

    @Override
    public Optional<Node> gameCreated(final Game<?, ?, ?, ?> game) {
        final Optional<
                Game<OthelloPlayer, OthelloState, OthelloMove, OthelloStrategy>> othelloGame = toOthelloGame(game);
        if (othelloGame.isPresent()) {
            final OthelloBoardView boardView = new OthelloBoardView(othelloGame.get());
            this.boardViews.put(game.getId(), boardView);
            return Optional.of(boardView.getNode());
        }
        return Optional.empty();
    }

    @Override
    public void gamePaused(final Game<?, ?, ?, ?> game) {
        final Optional<
                Game<OthelloPlayer, OthelloState, OthelloMove, OthelloStrategy>> othelloGame = toOthelloGame(game);
        if (othelloGame.isPresent()) {
            final OthelloBoardView boardView = this.boardViews.get(game.getId());
            if (boardView != null) {
                boardView.gamePaused(othelloGame.get());
            }
        }
    }

    @Override
    public void gameResumed(final Game<?, ?, ?, ?> game) {
        final Optional<
                Game<OthelloPlayer, OthelloState, OthelloMove, OthelloStrategy>> othelloGame = toOthelloGame(game);
        if (othelloGame.isPresent()) {
            final OthelloBoardView boardView = this.boardViews.get(game.getId());
            if (boardView != null) {
                boardView.gameResumed(othelloGame.get());
            }
        }
    }

    @Override
    public void gameDestroyed(final Game<?, ?, ?, ?> game) {
        final Optional<
                Game<OthelloPlayer, OthelloState, OthelloMove, OthelloStrategy>> othelloGame = toOthelloGame(game);
        if (othelloGame.isPresent()) {
            final OthelloBoardView boardView = this.boardViews.remove(game.getId());
            if (boardView != null) {
                boardView.destroy(othelloGame.orElseThrow());
            }
        }
    }

    /**
     * Casts the game to the right type if it is an Othello game. Otherwise an empty {@link Optional} is returned.
     *
     * @param game The game to check.
     */
    private static Optional<Game<OthelloPlayer, OthelloState, OthelloMove, OthelloStrategy>> toOthelloGame(
            final Game<?, ?, ?, ?> game) {
        if (game.getState() instanceof OthelloState) {
            @SuppressWarnings("unchecked")
            final Game<OthelloPlayer, OthelloState, OthelloMove, OthelloStrategy> othelloGame = (Game<OthelloPlayer,
                    OthelloState, OthelloMove, OthelloStrategy>) game;
            return Optional.of(othelloGame);
        }
        return Optional.empty();
    }
}
