/*
 * Copyright © 2020-2023 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of gaming-contest.
 *
 * Gaming-contest is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * Gaming-contest is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * gaming-contest. If not, see <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.contest;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.EnumMap;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.stream.Collectors;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.DefaultParser;
import org.apache.commons.cli.Options;

import de.fhdw.gaming.contest.ui.InteractiveStreamInputProvider;
import de.fhdw.gaming.contest.util.CombinatoricsHelper;
import de.fhdw.gaming.contest.util.ComparablePair;
import de.fhdw.gaming.core.domain.DefaultGameBuilderFactoryProvider;
import de.fhdw.gaming.core.domain.Game;
import de.fhdw.gaming.core.domain.GameBuilder;
import de.fhdw.gaming.core.domain.GameBuilderFactory;
import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.Player;
import de.fhdw.gaming.core.domain.PlayerState;
import de.fhdw.gaming.core.domain.Strategy;
import de.fhdw.gaming.core.domain.util.GameBuilderFactoryWrapper;
import de.fhdw.gaming.core.ui.InputProvider;
import de.fhdw.gaming.core.ui.InputProviderException;
import de.fhdw.gaming.core.ui.type.validator.MaxValueValidator;
import de.fhdw.gaming.core.ui.type.validator.MinValueValidator;
import de.fhdw.gaming.core.ui.util.ChainedInputProvider;
import de.fhdw.gaming.core.ui.util.NonInteractiveInputProvider;

/**
 * Provides the main entry point.
 */
public final class Main {

    /**
     * Type of game.
     */
    private static final String PARAM_GAME = "game";
    /**
     * The number of games to be played.
     */
    private static final String PARAM_REPEAT_COUNT = "repeatCount";
    /**
     * Do strategies play against themselves?
     */
    private static final String PARAM_PLAY_AGAINST_SELF = "playAgainstSelf";
    /**
     * Are strategies permutated?
     */
    private static final String PARAM_PERMUTATE_STRATEGIES = "permutateStrategies";

    /**
     * If non-null, the events of all contests are recorded into this directory.
     */
    private final File recordingDirectory;

    /**
     * Encapsulates a strategy together with its game counters.
     */
    private static final class StrategyData {

        /**
         * The name of the strategy.
         */
        private final String name;
        /**
         * Counts the number of games for each possible outcome.
         */
        private final EnumMap<PlayerState, Integer> countersPerState;
        /**
         * Denotes the total outcome for this strategy.
         */
        private double totalOutcome;

        /**
         * Constructor. The total outcome is initially set to zero.
         *
         * @param name             The name of the strategy.
         * @param countersPerState Counts the number of games for each possible player
         *                         state.
         */
        StrategyData(final String name, final EnumMap<PlayerState, Integer> countersPerState) {
            this.name = name;
            this.countersPerState = countersPerState;
            this.totalOutcome = 0.0;
        }

        /**
         * Returns the name of the strategy.
         */
        String getName() {
            return this.name;
        }

        /**
         * Returns the game counters for each possible player state.
         */
        EnumMap<PlayerState, Integer> getCounters() {
            return this.countersPerState;
        }

        /**
         * Returns the strategy's total outcome.
         */
        double getTotalOutcome() {
            return this.totalOutcome;
        }

        /**
         * Adds an outcome to the strategy's total outcome.
         *
         * @param outcomeToAdd The outcome to add.
         */
        void addOutcome(final double outcomeToAdd) {
            this.totalOutcome += outcomeToAdd;
        }
    }

    /**
     * Encapsulates the parameters of a competition. A competition consists of at
     * least one contest. Two contests of the same competition differ in the players
     * (and, hence, the strategies) involved. In our case, running a competition
     * will let players of each available strategy combination play against each
     * other.
     */
    private static final class CompetitionParameters {

        /**
         * {@link GameBuilderFactory} to use.
         */
        private final GameBuilderFactory gameBuilderFactory;
        /**
         * The number of games to be played per contest.
         */
        private final int numberOfGamesPerContest;
        /**
         * The maximum computation time in seconds per player and move.
         */
        private final int maxComputationTimePerMove;
        /**
         * {@code true} iff strategies should play against themselves.
         */
        private final boolean playAgainstSelf;
        /**
         * {@code true} iff strategies should be permutated.
         */
        private final boolean permutateStrategies;

        /**
         * Creates a {@link CompetitionParameters} object.
         *
         * @param gameBuilderFactory        The {@link GameBuilderFactory} to use.
         * @param numberOfGamesPerContest   The number of games per contest to be
         *                                  played.
         * @param maxComputationTimePerMove The maximum computation time in seconds per
         *                                  player and move.
         * @param playAgainstSelf           {@code true} iff strategies should play against themselves.
         * @param permutateStrategies       {@code true} iff strategies should be permutated.
         */
        CompetitionParameters(final GameBuilderFactory gameBuilderFactory, final int numberOfGamesPerContest,
                final int maxComputationTimePerMove, final boolean playAgainstSelf,
                final boolean permutateStrategies) {
            this.gameBuilderFactory = gameBuilderFactory;
            this.numberOfGamesPerContest = numberOfGamesPerContest;
            this.maxComputationTimePerMove = maxComputationTimePerMove;
            this.playAgainstSelf = playAgainstSelf;
            this.permutateStrategies = permutateStrategies;
        }

        /**
         * Returns the {@link GameBuilderFactory} to use.
         */
        GameBuilderFactory getGameBuilderFactory() {
            return this.gameBuilderFactory;
        }

        /**
         * Returns the number of games per contest to be played .
         */
        int getNumberOfGamesPerContest() {
            return this.numberOfGamesPerContest;
        }

        /**
         * Returns the maximum computation time in seconds per player and move.
         */
        public int getMaxComputationTimePerMove() {
            return this.maxComputationTimePerMove;
        }

        /**
         * Returns {@code true} iff strategies should play against themselves.
         */
        public boolean isPlayAgainstSelf() {
            return this.playAgainstSelf;
        }

        /**
         * Returns {@code true} iff strategies should be permutated.
         */
        public boolean isPermutateStrategies() {
            return this.permutateStrategies;
        }
    }

    /**
     * Encapsulates the results of a competition.
     */
    private static final class CompetitionResults {

        /**
         * The results of all games played.
         */
        private final Map<Class<?>, StrategyData> gameResults;
        /**
         * Total number of games played.
         */
        private int totalNumberOfGamesPlayed;

        /**
         * Creates a {@link CompetitionResults} object.
         */
        CompetitionResults() {
            this.gameResults = new LinkedHashMap<>();
        }

        /**
         * Returns the game results.
         */
        Map<Class<?>, StrategyData> getGameResults() {
            return this.gameResults;
        }

        /**
         * Returns the total number of games played.
         */
        int getTotalNumberOfGamesPlayed() {
            return this.totalNumberOfGamesPlayed;
        }

        /**
         * Increments the total number of games played.
         */
        void incrementTotalNumberOfGamesPlayed() {
            ++this.totalNumberOfGamesPlayed;
        }
    }

    /**
     * Encapsulates the parameters of a contest. A contest is a series of matches (=
     * games) between a single set of players involved, i.e. where the strategies
     * are kept fixed.
     */
    private static final class ContestParameters {

        /**
         * {@link GameBuilder} to use.
         */
        private final GameBuilder gameBuilder;
        /**
         * The number of games to be played.
         */
        private final int numberOfGames;

        /**
         * Creates a {@link ContestParameters} object.
         *
         * @param gameBuilder   The {@link GameBuilder} to use.
         * @param numberOfGames The number of games to be played.
         */
        ContestParameters(final GameBuilder gameBuilder, final int numberOfGames) {
            this.gameBuilder = gameBuilder;
            this.numberOfGames = numberOfGames;
        }

        /**
         * Returns the {@link GameBuilder} to use.
         */
        GameBuilder getGameBuilder() {
            return this.gameBuilder;
        }

        /**
         * Returns the number of games to be played.
         */
        int getNumberOfGames() {
            return this.numberOfGames;
        }
    }

    /**
     * Constructor.
     *
     * @param recordingDirectory If non-null, the events of all contests are
     *                           recorded into this directory.
     */
    public Main(final String recordingDirectory) {
        this.recordingDirectory = recordingDirectory != null ? new File(recordingDirectory) : null;
    }

    /**
     * Runs the contest.
     */
    public void run() throws InputProviderException, GameException, InterruptedException, IOException {
        final CompetitionParameters competitionParameters = this.determineCompetitionParameters();
        final CompetitionResults competitionResults = this.runCompetition(competitionParameters);

        System.out.println();

        final int numberOfGamesPlayed = competitionResults.getTotalNumberOfGamesPlayed();
        System.out.printf("A total of %d games played. Results:%n", numberOfGamesPlayed);

        this.printResult(numberOfGamesPlayed, competitionResults.getGameResults());
    }

    /**
     * Prints the result of the competition.
     *
     * @param numberOfGamesPlayed The number of games played.
     * @param gameResults         The game results.
     */
    private void printResult(final int numberOfGamesPlayed, final Map<Class<?>, StrategyData> gameResults) {
        final SortedMap<ComparablePair<Integer, Integer>, List<StrategyData>> ranking = new TreeMap<>();
        final Comparator<Integer> reverseCompare = ((Comparator<Integer>) Integer::compareTo).reversed();

        int maxNameLength = 0;
        int maxOutcomeLength = 0;
        for (final StrategyData strategyData : gameResults.values()) {
            final String name = strategyData.getName();
            // first order by games won, then (giving equality) by games that ended in a
            // draw
            final ComparablePair<Integer, Integer> key = ComparablePair.of(
                    strategyData.getCounters().getOrDefault(PlayerState.WON, 0), reverseCompare,
                    strategyData.getCounters().getOrDefault(PlayerState.DRAW, 0), reverseCompare);
            ranking.computeIfAbsent(key, (final var k) -> new ArrayList<>()).add(strategyData);
            maxNameLength = Math.max(maxNameLength, name.length());
            maxOutcomeLength = Math.max(maxOutcomeLength,
                    String.format("%.3f", strategyData.getTotalOutcome()).length());
        }

        final int maxNumFieldWidth = String.valueOf(numberOfGamesPlayed).length();

        int place = 1;
        for (final Map.Entry<ComparablePair<Integer, Integer>, List<StrategyData>> rankingEntry : ranking.entrySet()) {
            final List<StrategyData> strategyDataList = rankingEntry.getValue();
            boolean first = true;
            for (final StrategyData strategyData : strategyDataList) {
                System.out
                        .printf(
                                String.format(
                                        "  %%s Strategy %%-%1$ds "
                                                + "[WON: %%%2$dd, DRAW: %%%2$dd, LOST: %%%2$dd, RESIGNED: %%%2$dd] "
                                                + "[OUTCOME: %%%3$d.3f]%n",
                                        maxNameLength, maxNumFieldWidth, maxOutcomeLength),
                                String.format(first ? "%4d. place:" : "            ", place), strategyData.getName(),
                                rankingEntry.getKey().getFirst(), rankingEntry.getKey().getSecond(),
                                strategyData.getCounters().getOrDefault(PlayerState.LOST, 0),
                                strategyData.getCounters().getOrDefault(PlayerState.RESIGNED, 0),
                                strategyData.getTotalOutcome());
                first = false;
            }
            place += strategyDataList.size();
        }
    }

    /**
     * Determines the competition parameters.
     */
    private CompetitionParameters determineCompetitionParameters() throws InputProviderException {
        final Set<GameBuilderFactory> gameBuilderFactories = new LinkedHashSet<>();
        for (final GameBuilderFactory gameBuilderFactory : new DefaultGameBuilderFactoryProvider()
                .getGameBuilderFactories()) {
            gameBuilderFactories.add(new GameBuilderFactoryWrapper(gameBuilderFactory));
        }

        final InputProvider inputProvider = new InteractiveStreamInputProvider(System.in, System.out)
                .needObject(Main.PARAM_GAME, "Which game to play", Optional.empty(), gameBuilderFactories)
                .needInteger(Main.PARAM_REPEAT_COUNT, "Number of games per contest", Optional.of(100),
                        new MinValueValidator<>(1), new MaxValueValidator<>(1000))
                .needInteger(GameBuilderFactory.PARAM_MAX_COMPUTATION_TIME_PER_MOVE,
                        "Maximum computation time in seconds per player and move",
                        Optional.of(GameBuilder.DEFAULT_MAX_COMPUTATION_TIME_PER_MOVE), new MinValueValidator<>(1))
                .needBoolean(Main.PARAM_PLAY_AGAINST_SELF, "Do strategies play against themselves?",
                        Optional.of(false))
                .needBoolean(Main.PARAM_PERMUTATE_STRATEGIES, "Are strategies permutated?",
                        Optional.of(true));

        if (gameBuilderFactories.size() == 1) {
            inputProvider.fixedObject(Main.PARAM_GAME, gameBuilderFactories.iterator().next());
        }
        final Map<String, Object> data = inputProvider.requestData("Competition parameters");

        final GameBuilderFactory gameBuilderFactory = (GameBuilderFactory) data.get(Main.PARAM_GAME);
        final int repeatCount = (Integer) data.get(Main.PARAM_REPEAT_COUNT);
        final int maxComputationTimePerMove = (Integer) data
                .get(GameBuilderFactory.PARAM_MAX_COMPUTATION_TIME_PER_MOVE);
        final boolean playAgainstSelf = (Boolean) data.get(Main.PARAM_PLAY_AGAINST_SELF);
        final boolean permutateStrategies = (Boolean) data.get(Main.PARAM_PERMUTATE_STRATEGIES);
        return new CompetitionParameters(
                gameBuilderFactory, repeatCount, maxComputationTimePerMove, playAgainstSelf, permutateStrategies);
    }

    /**
     * Runs a competition.
     *
     * @param competitionParameters The competition parameters.
     * @return A map where to store how the strategies have performed in the
     *         competition.
     */
    private CompetitionResults runCompetition(final CompetitionParameters competitionParameters)
            throws InputProviderException, GameException, InterruptedException, IOException {

        final CompetitionResults competitionResults = new CompetitionResults();

        final GameBuilderFactory gameBuilderFactory = competitionParameters.getGameBuilderFactory();
        final List<Strategy<?, ?, ?>> strategies = new ArrayList<>(gameBuilderFactory.getStrategies());
        // a competition must run non-interactively, so we have to filter out
        // interactive strategies
        strategies.removeIf(Strategy::isInteractive);

        final Comparator<Strategy<?, ?, ?>> comp = Comparator.comparingInt(strategies::indexOf);

        int contestNumber = 1;
        for (int numPlayers = gameBuilderFactory.getMinimumNumberOfPlayers(); numPlayers <= gameBuilderFactory
                .getMaximumNumberOfPlayers(); ++numPlayers) {

            final List<List<Strategy<?, ?, ?>>> strategySets;
            if (competitionParameters.isPlayAgainstSelf()) {
                if (competitionParameters.isPermutateStrategies()) {
                    strategySets = CombinatoricsHelper
                            .combinationsWithRepetition(strategies, numPlayers)
                            .flatMap((final List<Strategy<?, ?, ?>> strategyList) -> CombinatoricsHelper
                                    .permutationsWithRepetition(strategyList, comp))
                            .collect(Collectors.toList());
                } else {
                    strategySets = CombinatoricsHelper
                            .combinationsWithRepetition(strategies, numPlayers)
                            .collect(Collectors.toList());
                }
            } else {
                if (competitionParameters.isPermutateStrategies()) {
                    strategySets = CombinatoricsHelper
                            .combinations(strategies, numPlayers)
                            .flatMap(CombinatoricsHelper::permutations)
                            .collect(Collectors.toList());
                } else {
                    strategySets = CombinatoricsHelper
                            .combinations(strategies, numPlayers)
                            .collect(Collectors.toList());
                }
            }

            for (final List<Strategy<?, ?, ?>> strategySet : strategySets) {
                System.out.printf("Contest %d. Number of players: %d. Strategies involved: %s%n",
                        contestNumber, numPlayers, strategySet);

                final ContestParameters contestParameters = this.determineContestParameters(competitionParameters,
                        strategySet);
                this.runContest(contestNumber, contestParameters, competitionResults);

                ++contestNumber;
            }
        }

        return competitionResults;
    }

    /**
     * Determines the contest parameters.
     *
     * @param competitionParameters The competition parameters.
     * @param strategies            A set of strategies to be used for this contest.
     */
    private ContestParameters determineContestParameters(final CompetitionParameters competitionParameters,
            final List<? extends Strategy<?, ?, ?>> strategies) throws InputProviderException, GameException {

        final ListIterator<? extends Strategy<?, ?, ?>> itStrategy = strategies.listIterator(strategies.size());

        InputProvider mostRecentProvider = null;
        int nextPlayerId = strategies.size();
        while (itStrategy.hasPrevious()) {
            final InputProvider lastPlayerProvider = mostRecentProvider;
            final InputProvider playerInputProvider = lastPlayerProvider == null ? new NonInteractiveInputProvider()
                    : new ChainedInputProvider(new NonInteractiveInputProvider(),
                            (final Map<String, Object> lastDataSet) -> lastPlayerProvider);

            playerInputProvider.fixedString(GameBuilderFactory.PARAM_PLAYER_NAME, Integer.toString(nextPlayerId--));
            playerInputProvider.fixedObject(GameBuilderFactory.PARAM_PLAYER_STRATEGY, itStrategy.previous());
            mostRecentProvider = playerInputProvider;
        }

        final InputProvider firstPlayerProvider = mostRecentProvider;
        final InputProvider gameInputProvider = firstPlayerProvider == null ? new NonInteractiveInputProvider()
                : new ChainedInputProvider(new NonInteractiveInputProvider(),
                        (final Map<String, Object> lastDataSet) -> firstPlayerProvider);
        gameInputProvider.fixedInteger(GameBuilderFactory.PARAM_MAX_COMPUTATION_TIME_PER_MOVE,
                competitionParameters.getMaxComputationTimePerMove());

        final GameBuilder gameBuilder = competitionParameters.getGameBuilderFactory()
                .createGameBuilder(gameInputProvider);
        return new ContestParameters(gameBuilder, competitionParameters.getNumberOfGamesPerContest());
    }

    /**
     * Runs a contest. If a recording directory has been configured, a
     * {@link RecordingObserver} is attached to each game in the contest in order to
     * log the game events to a file.
     *
     * @param contestNumber      The number of the contest.
     * @param contestParameters  The contest parameters.
     * @param competitionResults The competition results where to store how the
     *                           strategies have performed in the competition.
     */
    @SuppressWarnings("PMD.UseTryWithResources")
    private void runContest(final int contestNumber, final ContestParameters contestParameters,
            final CompetitionResults competitionResults) throws GameException, InterruptedException, IOException {

        final File contestDirectory;
        if (this.recordingDirectory != null) {
            contestDirectory = new File(this.recordingDirectory, String.format("%d", contestNumber));
        } else {
            contestDirectory = null;
        }

        final GameBuilder gameBuilder = contestParameters.getGameBuilder();

        for (int i = 1; i <= contestParameters.getNumberOfGames(); ++i) {
            RecordingObserver observer = null;
            try (Game<?, ?, ?, ?> game = gameBuilder.build(i)) {
                if (contestDirectory != null) {
                    final File gameEventFile = createGameEventFile(contestDirectory, String.format("%d.log", i));
                    observer = new RecordingObserver(gameEventFile);
                    game.addObserver(observer);
                }

                this.runGame(game, competitionResults);
            } finally {
                if (observer != null) {
                    observer.close();
                }
            }
        }

        System.out.println();
    }

    /**
     * Runs a game.
     *
     * @param game               The game to run.
     * @param competitionResults The competition results where to store how the
     *                           strategies have performed in the competition.
     */
    private void runGame(final Game<?, ?, ?, ?> game, final CompetitionResults competitionResults)
            throws InterruptedException {
        final Map<Class<?>, StrategyData> gameResults = competitionResults.getGameResults();

        game.start();

        while (!game.isFinished()) {
            game.makeMove();
        }

        competitionResults.incrementTotalNumberOfGamesPlayed();
        System.out.print('.');

        final Map<String, ? extends Strategy<?, ?, ?>> strategies = game.getStrategies();
        for (final Player<?> player : game.getPlayers().values()) {
            final Strategy<?, ?, ?> strategy = strategies.get(player.getName());
            final Class<?> strategyClass = strategy.getClass();

            final StrategyData strategyData = gameResults.computeIfAbsent(strategyClass,
                    (final var k) -> new StrategyData(strategy.toString(), new EnumMap<>(PlayerState.class)));
            strategyData.getCounters().compute(player.getState(),
                    (final PlayerState key, final Integer value) -> value == null ? 1 : value + 1);
            strategyData.addOutcome(player.getOutcome().orElseThrow());
        }
    }

    /**
     * Returns a {@link File} object for storing game events.
     *
     * @param parentDirectory The parent directory. May be {@code null} if game
     *                        events shall not be saved.
     * @param fileName        The file name.
     * @return An appropriate {@link File} object or {@code null} if game events
     *         shall not be saved.
     * @throws IOException if creating the parent directory (if not present yet) has
     *                     failed.
     */
    private static File createGameEventFile(final File parentDirectory, final String fileName) throws IOException {
        if (parentDirectory == null) {
            return null;
        }

        if (!parentDirectory.exists() && !parentDirectory.mkdirs()) {
            throw new IOException(String.format("Could not create directory for recording events: %s",
                    parentDirectory.getCanonicalPath()));
        }
        if (!parentDirectory.isDirectory()) {
            throw new IOException(String.format("Directory for recording events is not a directory: %s",
                    parentDirectory.getCanonicalPath()));
        }

        return new File(parentDirectory, fileName);
    }

    /**
     * The main entry point.
     *
     * @param parameters The parameters passed to the application.
     */
    public static void main(final String[] parameters) {
        final Options options = new Options();
        options.addOption("r", "recordDir", true, "Specifies the directory where to record events");

        try {
            final CommandLine commandLine = new DefaultParser().parse(options, parameters);
            new Main(commandLine.getOptionValue('r')).run();
        } catch (final Exception e) {
            System.err.print("Game contest aborted due to exception: ");
            e.printStackTrace(System.err);
            System.exit(3);
        }
    }
}
