package de.fhdw.gaming.contest.util;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.LinkedList;
import java.util.List;
import java.util.stream.LongStream;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

/**
 * Contains methods for operations belonging to combinatorics .
 */
public final class CombinatoricsHelper {

    /**
     * Private constructor as this is a helper class.
     */
    private CombinatoricsHelper() {
    }

    /**
     * Computes the factorial of a given number.
     *
     * @param num The number.
     * @return The corresponding factorial. If the argument is less than one, the result is 1.
     */
    public static long factorial(final int num) {
        return nthProduct(1, num);
    }

    /**
     * Computes the product between two values.
     *
     * @param start The start value (exclusive).
     * @param end   The end value (inclusive).
     * @return The corresponding factorial. If start &lt; end, the result is 1.
     */
    public static long nthProduct(final int start, final int end) {
        return LongStream.rangeClosed(start + 1, end).reduce(1L, (final long x, final long y) -> x * y);
    }

    /**
     * Computes the binomial coefficient n-over-k.
     * <p>
     * The implementation is optimised in such a way that large intermediate results are avoided.
     *
     * @param nValue The "n" value.
     * @param kValue The "k" value.
     * @return The corresponding binomial coefficient n-over-k. If n &lt; k, the result is zero.
     */
    public static long nOverK(final int nValue, final int kValue) {
        if (nValue < kValue) {
            return 0;
        } else if (kValue < nValue - kValue) {
            return nthProduct(nValue - kValue, nValue) / factorial(kValue);
        } else {
            return nthProduct(kValue, nValue) / factorial(nValue - kValue);
        }
    }

    /**
     * Returns a stream of permutations of a given list without repetition.
     *
     * @see <a href= "https://gist.github.com/jaycobbcruz/cbabc1eb49f51bfe2ed1db10a06a2b26">
     *      https://gist.github.com/jaycobbcruz/cbabc1eb49f51bfe2ed1db10a06a2b26</a>
     *
     * @param <T>   The type of items in the list.
     * @param items The list of items.
     * @return A stream of permutations of the given list.
     */
    public static <T> Stream<List<T>> permutations(final List<T> items) {
        return LongStream.range(0L, factorial(items.size()))
                .mapToObj((final long permutationNumber) -> permutation(permutationNumber, new LinkedList<>(items)));
    }

    /**
     * Returns a stream of permutations of a given list.
     *
     * @param <T>        The type of items in the list.
     * @param items      The list of items.
     * @param comparator The comparator to use. If {@code null}, the items must implement the {@link Comparable}
     *                   interface.
     * @return A stream of permutations of the given list.
     */
    public static <T> Stream<List<T>> permutationsWithRepetition(final List<T> items, final Comparator<T> comparator) {
        return StreamSupport.stream(new PermutationSplitIterator<>(items, comparator), false);
    }

    /**
     * Returns a stream of permutations of a given list.
     *
     * @param <T>   The type of items in the list.
     * @param items The list of items.
     * @return A stream of permutations of the given list.
     */
    public static <T extends Comparable<T>> Stream<List<T>> permutationsWithRepetition(final List<T> items) {
        return StreamSupport.stream(PermutationSplitIterator.create(items), false);
    }

    /**
     * Returns a stream of combinations of a given list without repetition.
     *
     * @param <T>         The type of items in the list.
     * @param items       The list of items.
     * @param sublistSize The size of the combinations.
     * @return A stream of suitable combinations over the given list. Note that the stream will be empty if the size of
     *         the combinations is greater than the size of the list of items.
     */
    public static <T> Stream<List<T>> combinations(final List<T> items, final int sublistSize) {
        return StreamSupport.stream(new CombinationSplitIterator<>(items, sublistSize, false), false);
    }

    /**
     * Returns a stream of combinations of a given list with repetition.
     *
     * @param <T>         The type of items in the list.
     * @param items       The list of items.
     * @param sublistSize The size of the combinations.
     * @return A stream of suitable combinations over the given list. Note that the stream will be empty if and only if
     *         the list of items is empty.
     */
    public static <T> Stream<List<T>> combinationsWithRepetition(final List<T> items, final int sublistSize) {
        return StreamSupport.stream(new CombinationSplitIterator<>(items, sublistSize, true), false);
    }

    /**
     * Computes a certain permutation of a given list without repetition.
     *
     * @see <a href= "https://gist.github.com/jaycobbcruz/cbabc1eb49f51bfe2ed1db10a06a2b26">
     *      https://gist.github.com/jaycobbcruz/cbabc1eb49f51bfe2ed1db10a06a2b26</a>
     *
     * @param <T>               The type of items in the list.
     * @param permutationNumber The number of the permutation to compute.
     * @param input             The list of items.
     * @return The corresponding permutation of the list.
     */
    private static <T> List<T> permutation(final long permutationNumber, final List<T> input) {
        long count = permutationNumber;
        long factorial = factorial(input.size());
        final List<T> output = new ArrayList<>();
        while (!input.isEmpty()) {
            factorial /= input.size();
            final long index = count / factorial;
            assert (int) index == index; // detect arithmetic overflow
            output.add(input.remove((int) index));
            count %= factorial;
        }
        return output;
    }
}
