/*
 * Copyright © 2020-2023 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of gaming-gui.
 *
 * Gaming-gui is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * Gaming-gui is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * gaming-gui. If not, see <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.gui;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.Semaphore;
import java.util.concurrent.atomic.AtomicBoolean;

import de.fhdw.gaming.core.domain.Game;
import de.fhdw.gaming.gui.util.FXUtil;
import javafx.application.Platform;
import javafx.beans.property.ReadOnlyBooleanProperty;
import javafx.beans.property.ReadOnlyBooleanWrapper;
import javafx.scene.Node;
import javafx.scene.control.Alert.AlertType;
import javafx.scene.layout.GridPane;
import javafx.scene.layout.Priority;

/**
 * Runs the game. A game can be run at most once. Should the need arise to run a game a second time, a new
 * {@link GameRunner} object must be created and used.
 */
final class GameRunner implements Runnable {

    /**
     * The registered GUI observers.
     */
    private final List<GuiObserver> observers;
    /**
     * The game.
     */
    private final Optional<Game<?, ?, ?, ?>> game;
    /**
     * The game pane.
     */
    private final GridPane gamePane;
    /**
     * The thread running the game.
     */
    private final Thread thread;
    /**
     * Semaphore to wait on in order to run the game and to make moves.
     */
    private final Semaphore gameSemaphore;
    /**
     * Semaphore to wait on in order to terminate.
     */
    private final Semaphore exitSemaphore;
    /**
     * {@code true} if game should pause, else {@code false}.
     */
    private final AtomicBoolean pauseRequested;
    /**
     * {@code true} if game should be cancelled, else {@code false}.
     */
    private final AtomicBoolean abortRequested;
    /**
     * Property that indicates whether the game is running.
     */
    private final ReadOnlyBooleanWrapper running;
    /**
     * Property that indicates whether the game is over.
     */
    private final ReadOnlyBooleanWrapper finished;

    /**
     * Creates a {@link GameRunner}. The ownership of the game is passed to the {@link GameRunner} which is responsible
     * to close it at the very end. The game is initially paused and has to be started by calling
     * {@link #continueGame()}.
     *
     * @param observers The registered GUI observers.
     * @param game      The game.
     * @param gamePane  The game pane.
     */
    GameRunner(final List<GuiObserver> observers, final Game<?, ?, ?, ?> game, final GridPane gamePane) {
        this.observers = observers;
        this.game = Optional.of(game);
        this.gamePane = gamePane;
        this.gameSemaphore = new Semaphore(0);
        this.exitSemaphore = new Semaphore(0);
        this.pauseRequested = new AtomicBoolean(true);
        this.abortRequested = new AtomicBoolean(false);
        this.running = new ReadOnlyBooleanWrapper(false);
        this.finished = new ReadOnlyBooleanWrapper(false);
        this.thread = new Thread(this);
    }

    /**
     * Returns a property that indicates whether the game is running.
     */
    ReadOnlyBooleanProperty runningProperty() {
        return this.running.getReadOnlyProperty();
    }

    /**
     * Returns a property that indicates whether the game is over.
     */
    ReadOnlyBooleanProperty finishedProperty() {
        return this.finished.getReadOnlyProperty();
    }

    /**
     * Starts the execution of the game in a separate thread. The game is initially paused, i.e. execution stops before
     * the first move.
     */
    void startGame() {
        this.thread.start();
    }

    /**
     * Starts or continues the game.
     */
    void continueGame() {
        this.game.orElseThrow(() -> new IllegalStateException("GameRunner already closed, reuse impossible!"));
        this.pauseRequested.set(false);
        this.running.set(true);
        this.gameSemaphore.release();
    }

    /**
     * Pauses the game.
     */
    void pauseGame() {
        this.game.orElseThrow(() -> new IllegalStateException("GameRunner already closed, reuse impossible!"));
        this.pauseRequested.set(true);
        this.running.set(false);
    }

    /**
     * Aborts the game.
     */
    void abortGame() {
        this.game.orElseThrow(() -> new IllegalStateException("GameRunner already closed, reuse impossible!"));
        this.abortRequested.set(true);
        this.game.get().abortRequested();
        // make it work even if the game is currently paused
        this.pauseRequested.set(false);
        this.gameSemaphore.release();
    }

    /**
     * Terminates the {@link GameRunner}. Requires the game to have already finished.
     */
    void terminate() {
        this.exitSemaphore.release();
        try {
            this.thread.join();
        } catch (final InterruptedException e) {
            // return
        }
    }

    /**
     * Runs a game. Calls {@link Game#close()} in any case at the end of execution.
     */
    @Override
    public void run() {
        try {
            try {
                final List<Node> nodes = new ArrayList<>();
                this.observers.forEach((final GuiObserver o) -> {
                    final Optional<Node> node = o.gameCreated(this.game.orElseThrow());
                    node.ifPresent(nodes::add);
                });

                this.game.orElseThrow().start();
                Platform.runLater(() -> this.layoutNodes(nodes));

                while (!this.game.orElseThrow().isFinished()) {
                    this.pauseIfRequested();

                    if (this.abortRequested.get()) {
                        break;
                    }

                    this.game.orElseThrow().makeMove();
                }
            } finally {
                this.cleanupGame();
            }
        } catch (final InterruptedException e) {
            // return
        } catch (final Exception e) {
            Platform.runLater(() -> {
                FXUtil.showAlert(this.gamePane.getScene().getWindow(), AlertType.ERROR, e.getMessage());
                this.finished.set(true);
            });
        } finally {
            this.game.orElseThrow().close();
        }
    }

    /**
     * Pauses the game if requested.
     */
    private void pauseIfRequested() throws InterruptedException {
        while (this.pauseRequested.get()) {
            this.observers.forEach((final GuiObserver o) -> o.gamePaused(this.game.orElseThrow()));
            this.gameSemaphore.acquire();
            this.observers.forEach((final GuiObserver o) -> o.gameResumed(this.game.orElseThrow()));
        }
    }

    /**
     * Cleans up after game has finished.
     */
    private void cleanupGame() throws InterruptedException {
        Platform.runLater(() -> {
            this.running.set(false);
            this.finished.set(true);
        });
        this.exitSemaphore.acquire();
        this.observers.forEach((final GuiObserver o) -> o.gameDestroyed(this.game.orElseThrow()));
    }

    /**
     * Lays out the game nodes.
     *
     * @param nodes The nodes to lay out.
     */
    private void layoutNodes(final List<Node> nodes) {
        final int numColumns = (int) Math.ceil(Math.sqrt(nodes.size()));

        int currentRow = 0;
        int currentColumn = 0;

        for (final Node node : nodes) {
            this.gamePane.add(node, currentColumn, currentRow);
            GridPane.setHgrow(node, Priority.ALWAYS);
            GridPane.setVgrow(node, Priority.ALWAYS);
            if (currentRow * numColumns + currentColumn + 1 == nodes.size()) {
                GridPane.setColumnSpan(node, numColumns - currentColumn);
            }

            currentColumn = (currentColumn + 1) % numColumns;
            if (currentColumn == 0) {
                ++currentRow;
            }
        }
    }
}
